#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/load_units/moderation.h>

#include <maps/libs/chrono/include/days.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/robot.h>
#include <maps/wikimap/mapspro/libs/assessment/include/gateway.h>

#include <maps/wikimap/mapspro/libs/social/tests/helpers/event_creator.h>
#include <maps/wikimap/mapspro/libs/social/tests/helpers/task_creator.h>
#include <maps/wikimap/mapspro/libs/unittest/include/yandex/maps/wiki/unittest/arcadia.h>
#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/tests/helpers.h>

#include <library/cpp/testing/unittest/registar.h>

#define CHECK_UNIT_MATCHES_RESOLVED(unit_, task_)\
    do {\
        UNIT_ASSERT(task_.isResolved());\
        UNIT_ASSERT_EQUAL(unit_.entity.id, std::to_string(task_.id()));\
        UNIT_ASSERT_EQUAL(unit_.entity.domain, assessment::Entity::Domain::Moderation);\
        UNIT_ASSERT_EQUAL(unit_.action.name, "resolve");\
        UNIT_ASSERT_EQUAL(unit_.action.by, task_.resolved().uid());\
        UNIT_ASSERT_EQUAL(unit_.action.at, chrono::parseSqlDateTime(task_.resolved().date()));\
    } while (false)

#define CHECK_UNIT_MATCHES_CLOSED(unit_, task_)\
    do {\
        UNIT_ASSERT(task_.isClosed());\
        UNIT_ASSERT_EQUAL(unit_.entity.id, std::to_string(task_.id()));\
        UNIT_ASSERT_EQUAL(unit_.entity.domain, assessment::Entity::Domain::Moderation);\
        UNIT_ASSERT_EQUAL(unit_.action.name, "close");\
        UNIT_ASSERT_EQUAL(unit_.action.by, task_.closed().uid());\
        UNIT_ASSERT_EQUAL(unit_.action.at, chrono::parseSqlDateTime(task_.closed().date()));\
    } while (false)

namespace maps::wiki::assessment::sampler::tests {

using namespace std::chrono_literals;
using social::tests::EventCreator;
using social::tests::TaskCreator;

Y_UNIT_TEST_SUITE_F(load_units_moderation_tests, unittest::ArcadiaDbFixture) {
    Y_UNIT_TEST(should_collect_moderation_during_time_period) {
        auto txn = pool().masterWriteableTransaction();
        const auto [timepointMin, timepointMax] = txnNowSpan(*txn);

        const std::vector<social::Task> tasks = {
            createResolvedTask(*txn, USER, timepointMin - 1s),
            createClosedTask(*txn, USER, timepointMin - 1s, USER, timepointMin),
            createResolvedTask(*txn, USER, timepointMin),
            createClosedTask(*txn, USER, timepointMin + 1s, ANOTHER_USER, timepointMax - 1s),
            createClosedTask(*txn, USER, timepointMax - 1s, USER, timepointMax),
            createResolvedTask(*txn, USER, timepointMax + 1s)};

        social::Gateway socialGw{*txn};
        const auto groupNameToUnits = loadModerationUnits(socialGw, timepointMin, timepointMax, {USER, ANOTHER_USER});
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 2);

        auto userUnits = groupNameToUnits.at("common-1");
        sortUnits(userUnits);
        UNIT_ASSERT_EQUAL(userUnits.size(), 4);
        CHECK_UNIT_MATCHES_CLOSED  (userUnits[0], tasks[1]);
        CHECK_UNIT_MATCHES_RESOLVED(userUnits[1], tasks[2]);
        CHECK_UNIT_MATCHES_RESOLVED(userUnits[2], tasks[3]);
        CHECK_UNIT_MATCHES_RESOLVED(userUnits[3], tasks[4]);

        auto anotherUserUnits = groupNameToUnits.at("common-2");
        UNIT_ASSERT_EQUAL(anotherUserUnits.size(), 1);
        CHECK_UNIT_MATCHES_CLOSED(anotherUserUnits[0], tasks[3]);
    }

    Y_UNIT_TEST(should_group_moderation_by_category_resolved_by_and_closed_by) {
        auto txn = pool().masterWriteableTransaction();
        const auto [timepointMin, timepointMax] = txnNowSpan(*txn);

        TaskCreator(*txn).event(EventCreator(*txn).primaryObjData({11, "category_a", {}, {}})).resolved(1)();
        TaskCreator(*txn).event(EventCreator(*txn).primaryObjData({22, "category_a", {}, {}})).resolved(2).closed(1)();
        TaskCreator(*txn).event(EventCreator(*txn).primaryObjData({33, "category_b", {}, {}})).resolved(1).closed(2)();

        social::Gateway socialGw{*txn};
        const auto groupNameToUnits = loadModerationUnits(socialGw, timepointMin, timepointMax, {1, 2});
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 4);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("category_a-1").size(), 2);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("category_a-2").size(), 1);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("category_b-1").size(), 1);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("category_b-2").size(), 1);
    }

    Y_UNIT_TEST(should_ignore_moderation_by_robots) {
        auto txn = pool().masterWriteableTransaction();
        const auto [timepointMin, timepointMax] = txnNowSpan(*txn);

        ASSERT(!common::isRobot(USER) && !common::isRobot(ANOTHER_USER));
        createResolvedTask(*txn, USER, timepointMin + 1s);
        createClosedTask(*txn, USER, timepointMax - 2s, ANOTHER_USER, timepointMax -1s);

        for (const auto robotUid : common::ALL_ROBOTS_UIDS) {
            createResolvedTask(*txn, robotUid, timepointMin + 1s);
            createClosedTask(*txn, robotUid, timepointMax - 2s, robotUid, timepointMax - 1s);
        }

        auto allowedUids = social::TUids{common::ALL_ROBOTS_UIDS.cbegin(), common::ALL_ROBOTS_UIDS.cend()};
        allowedUids.insert(USER);
        allowedUids.insert(ANOTHER_USER);

        social::Gateway socialGw{*txn};
        const auto groupNameToUnits = loadModerationUnits(socialGw, timepointMin, timepointMax, allowedUids);
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 2);

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-1").size(), 2);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-2").size(), 1);

        for (const auto& [_, units]: groupNameToUnits) {
            for (const auto& unit: units) {
                UNIT_ASSERT(!common::isRobot(unit.action.by));
            }
        }
    }

    Y_UNIT_TEST(should_collect_sample_for_allowed_users_only)
    {
        auto txn = pool().masterWriteableTransaction();
        const auto [timepointMin, timepointMax] = txnNowSpan(*txn);

        const std::array taskIds = {
            std::to_string(createClosedTask(*txn, 1, timepointMin + 1h, 2, timepointMin + 2h).id()),
            std::to_string(createClosedTask(*txn, 3, timepointMin + 1h, 4, timepointMin + 2h).id()),
            std::to_string(createClosedTask(*txn, 5, timepointMin + 1h, 6, timepointMin + 2h).id())
        };

        social::Gateway socialGw{*txn};
        const auto groupNameToUnits = loadModerationUnits(socialGw, timepointMin, timepointMax, {2, 5});
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 2);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-2").size(), 1);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-5").size(), 1);

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-2")[0].entity.id, taskIds[0]);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("common-5")[0].entity.id, taskIds[2]);
    }
}

} // maps::wiki::assessment::sampler::tests
