#include <maps/wikimap/mapspro/services/tasks_social/src/assessment_sampler/lib/load_units/tracker.h>

#include <maps/libs/st/ut/helpers/issue_creator.h>
#include <maps/wikimap/mapspro/libs/assessment/include/unit.h>

#include <library/cpp/testing/unittest/registar.h>


#define CHECK_UNIT(unit_, entityId_, domain_, actionBy_, actionAt_, action_)     \
    do {                                                                         \
        UNIT_ASSERT_EQUAL(unit_.entity.id, entityId_);                           \
        UNIT_ASSERT_EQUAL(unit_.entity.domain, domain_);                         \
        UNIT_ASSERT_EQUAL(unit_.action.by, actionBy_);                           \
        UNIT_ASSERT_EQUAL(unit_.action.at, chrono::parseIsoDateTime(actionAt_)); \
        UNIT_ASSERT_STRINGS_EQUAL(unit_.action.name, action_);                   \
    } while (false)


namespace maps::wiki::assessment::sampler::tests {

using st::tests::IssueCreator;

Y_UNIT_TEST_SUITE(load_units_tracker_tests) {
    Y_UNIT_TEST(should_collect_tracker_sample)
    {
        const auto groupNameToUnits = loadTrackerUnits(
            {
                {"assignee 1", 1},
                {"assignee 2", 2}, // unused
                {"assignee 3", 3},
                {"assignee 4", 4},
                {"assignee 6", 6}
            },
            chrono::parseIsoDate("2021-10-18"),
            {
                *IssueCreator("QUEUE-1").resolvedAt("2021-10-18T01:01:01.000+0000").assigneeId("assignee 1"),
                *IssueCreator("QUEUE-3").resolvedAt("2021-10-18T03:03:03.000+0000").assigneeId("assignee 3")
            },
            {
                *IssueCreator("QUEUE-4").assigneeId("assignee 4"),
                *IssueCreator("QUEUE-6").assigneeId("assignee 6")
            }
        );
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 4);

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("close-1").size(), 1);
        CHECK_UNIT(groupNameToUnits.at("close-1")[0], "QUEUE-1", Entity::Domain::Tracker, 1, "2021-10-18T01:01:01.000+0000", "close");

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("close-3").size(), 1);
        CHECK_UNIT(groupNameToUnits.at("close-3")[0], "QUEUE-3", Entity::Domain::Tracker, 3, "2021-10-18T03:03:03.000+0000", "close");

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("change-status-to-on-support-side-4").size(), 1);
        CHECK_UNIT(groupNameToUnits.at("change-status-to-on-support-side-4")[0], "QUEUE-4", Entity::Domain::Tracker, 4, "2021-10-18T00:00:00.000+0000", "change-status-to-on-support-side");

        UNIT_ASSERT_EQUAL(groupNameToUnits.at("change-status-to-on-support-side-6").size(), 1);
        CHECK_UNIT(groupNameToUnits.at("change-status-to-on-support-side-6")[0], "QUEUE-6", Entity::Domain::Tracker, 6, "2021-10-18T00:00:00.000+0000", "change-status-to-on-support-side");
    }

    Y_UNIT_TEST(should_group_by_assignee_and_type) {
        const auto groupNameToUnits = loadTrackerUnits(
            {
                {"assignee 1", 1},
                {"assignee 2", 2},
            },
            chrono::parseIsoDate("2021-10-18"),
            {
                *IssueCreator("QUEUE-11").resolvedAt("2021-10-18T01:01:01.000+0000").assigneeId("assignee 1"),
                *IssueCreator("QUEUE-12").resolvedAt("2021-10-18T01:01:01.000+0000").assigneeId("assignee 2"),
                *IssueCreator("QUEUE-13").resolvedAt("2021-10-18T01:01:01.000+0000").assigneeId("assignee 2"),
            },
            {
                *IssueCreator("QUEUE-21").assigneeId("assignee 1"),
                *IssueCreator("QUEUE-22").assigneeId("assignee 1"),
                *IssueCreator("QUEUE-23").assigneeId("assignee 2"),
            }
        );
        UNIT_ASSERT_EQUAL(groupNameToUnits.size(), 4);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("close-1").size(), 1);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("close-2").size(), 2);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("change-status-to-on-support-side-1").size(), 2);
        UNIT_ASSERT_EQUAL(groupNameToUnits.at("change-status-to-on-support-side-2").size(), 1);
    }

    Y_UNIT_TEST(should_fail_collecting_tracker_sample_from_closed_tickets_without_resolved_at)
    {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            loadTrackerUnits(
                {{"assignee 1", 1}},
                chrono::parseIsoDate("2021-10-18"),
                {*IssueCreator("QUEUE-1").assigneeId("assignee 1")},
                {}
            ),
            RuntimeError,
            "Ticket 'QUEUE-1' has no resolution date."
        );
    }

    Y_UNIT_TEST(should_not_collect_tracker_sample_without_assignee)
    {
        const auto groupNameToUnits = loadTrackerUnits(
            {{"assignee 1", 1}},
            chrono::parseIsoDate("2021-10-18"),
            {*IssueCreator("QUEUE-1").resolvedAt("2021-10-18T01:01:01.000+0000")},
            {*IssueCreator("QUEUE-2")}
        );
        UNIT_ASSERT(groupNameToUnits.empty());
    }

    Y_UNIT_TEST(should_not_collect_tracker_sample_for_assignee_absent_in_staff_login_to_puid)
    {
        const auto groupNameToUnits = loadTrackerUnits(
            {},
            chrono::parseIsoDate("2021-10-18"),
            {*IssueCreator("QUEUE-1").resolvedAt("2021-10-18T01:01:01.000+0000").assigneeId("assignee 1")},
            {*IssueCreator("QUEUE-2").assigneeId("assignee 2")}
        );
        UNIT_ASSERT(groupNameToUnits.empty());
    }
}

} // maps::wiki::assessment::sampler::tests
