#include "addresses_email.h"

#include <yandex/maps/wiki/social/sent_notification.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/common/include/environment.h>

#include <chrono>

namespace maps::wiki::auto_mailer {

namespace {

const int64_t ADDRESSES_EMAIL_PERIOD_DAYS = 30;
const uint64_t MINIMAL_SHOWS_COUNT = 5;

const TString YT_ACCOUNT_DIR = "//home/maps/core/nmaps/";
const std::string ADDRESSES_YT_DIR =
    "/mailing/community/addresses/";
const std::string ADDRESSES_YT_TABLE_PREFIX = "shows_365_30_";


bool userWaitsMessage(
        const AddressesMessageInfo& messageInfo,
        social::SentNotificationGateway& notificationsGateway
) {
    if (messageInfo.showsCount() < MINIMAL_SHOWS_COUNT) {
        return false;
    }

    auto now = chrono::TimePoint::clock::now();
    auto notifications = notificationsGateway.load(
        social::table::SentNotificationTbl::uid == messageInfo.puid() &&
        social::table::SentNotificationTbl::type == social::NotificationType::AddressesShowsCount
    );

    using days = std::chrono::duration<int64_t, std::ratio<3600 * 24>>;

    for (const auto& notification : notifications) {
        auto sentAgo = now - notification.sentAt;
        auto sentDaysAgo = std::chrono::duration_cast<days>(sentAgo).count();

        if (sentDaysAgo < ADDRESSES_EMAIL_PERIOD_DAYS) {
            return false;
        }
    }

    return true;
}

TString addressesYtTable() {
    std::string envString;
    if (maps::common::getYandexEnvironment() == maps::common::Environment::Stable) {
        envString = "production";
    } else {
        envString = "testing";
    }

    auto yesterday = chrono::TimePoint::clock::now() - std::chrono::hours(24);
    return YT_ACCOUNT_DIR + envString + ADDRESSES_YT_DIR +
        ADDRESSES_YT_TABLE_PREFIX + chrono::formatIsoDate(yesterday);
}

std::vector<AddressesMessageInfo> loadAddressesDataFromYt(
    NYT::IClientPtr client,
    const auto& ytTable
) {
    std::vector<AddressesMessageInfo> result;

    INFO() << "Reading email data from table: " << ytTable;

    for (auto reader = client->CreateTableReader<NYT::TNode>(ytTable);
            reader->IsValid(); reader->Next()) {
        result.emplace_back(reader->GetRow());
    }

    return result;
}

}  // namespace

void sendAddressesMessages(
    NYT::IClientPtr client,
    EmailQueue& emailQueue,
    social::SentNotificationGateway& notificationsGateway
) {
    auto ytTable = addressesYtTable();
    if (!client->Exists(ytTable)) {
        WARN() << "Table does not exist: " << ytTable;
        return;
    }

    auto addressesData = loadAddressesDataFromYt(client, ytTable);
    for (const auto& messageInfo : addressesData) {
        if (userWaitsMessage(messageInfo, notificationsGateway)) {
            emailQueue.send(messageInfo);
        }
    }

    INFO() << "Removing table: " << ytTable;
    client->Remove(ytTable);
}

}  // namespace maps::wiki::auto_mailer
