// TODO(efrolov89): separate into several files: one for each involvement.
#include "filtration.h"
#include "utils.h"

#include "entrances_with_flats_counter.h"
#include "new_feedback_onfoot_counter.h"
#include "new_objects_with_ft_type_counter.h"

#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/social/involvement.h>

namespace rev = maps::wiki::revision;
namespace filter = maps::wiki::revision::filters;
using namespace maps::wiki::social;

namespace maps::wiki::tasks::involvement {

CommitsCounterFactory::CommitsCounterFactory(
    pqxx::transaction_base& coreTxn,
    pqxx::transaction_base& socialMasterTxn,
    pqxx::transaction_base& mrcReadTxn,
    const configs::editor::ConfigHolder& editorConfig)
    : coreTxn_(coreTxn)
    , socialMasterTxn_(socialMasterTxn)
    , mrcReadTxn_(mrcReadTxn)
    , editorConfig_(editorConfig)
{}

ICounterPtr
CommitsCounterFactory::createCounter(
    const std::string& involvementType,
    const std::vector<revision::DBID>& commitIds,
    std::optional<chrono::TimePoint> involvementStartOpt,
    std::optional<chrono::TimePoint> involvementFinishOpt,
    int64_t previousValue) const
{
    if (involvementType == involvement_name::SPEED_LIMITS) {
        return std::make_unique<ChangedRdElLengthCounter>(
            commitIds, coreTxn_, ATTR_SPEED_LIMIT, RdElChangeCountPolicy::AVOID_CHEATING);
    } else if (involvementType == involvement_name::UNPAVED_ROADS) {
        return std::make_unique<ChangedRdElLengthCounter>(
            commitIds, coreTxn_, ATTR_PAVED, RdElChangeCountPolicy::COUNT_ATTRIBUTE_MODIFICATION
        );
    } else if (involvementType == involvement_name::COND_TRAFFIC_LIGHT) {
        return std::make_unique<TrafficLightsCounter>(commitIds, coreTxn_);
    } else if (involvementType == involvement_name::PARKING_LOTS_LINEAR) {
        return std::make_unique<NewObjectsLengthCounter>(commitIds, coreTxn_, CAT_PARKING_LOTS_LINEAR);
    } else if (involvementType == involvement_name::COVID) {
        return std::make_unique<CovidCommitsCounter>(commitIds, coreTxn_, socialMasterTxn_, editorConfig_);
    } else if (involvementType == involvement_name::MRC_FEEDBACK_MIRRORS) {
        return std::make_unique<MirrorCoverageCounter>();
    } else if (involvementType == involvement_name::MRC_FEEDBACK_ONFOOT) {
        return std::make_unique<NewOnfootFeedbackCounter>(
            coreTxn_,
            mrcReadTxn_,
            involvementStartOpt,
            involvementFinishOpt,
            previousValue);
    } else if (involvementType == involvement_name::FLAT_RANGE) {
        return std::make_unique<EntrancesWithFlatsCounter>(commitIds, coreTxn_);
    } else if (involvementType == involvement_name::CREATED_FENCES_LENGTH) {
        return std::make_unique<NewObjectsLengthCounter>(commitIds, coreTxn_, CAT_URBAN_ROADNET_FENCE_EL);
    } else if (involvementType == involvement_name::CREATED_BENCHES) {
        return std::make_unique<NewObjectsWithFtTypeCounter>(
            commitIds,
            coreTxn_,
            CAT_POI_URBAN,
            std::set<std::string>{FT_TYPE_URBAN_BENCH});
    } else if (involvementType == involvement_name::CREATED_BARBEQUE) {
        return std::make_unique<NewObjectsWithFtTypeCounter>(
            commitIds, coreTxn_,
            CAT_POI_URBAN,
            std::set<std::string>{FT_TYPE_URBAN_BARBEQUE});
    } else if (involvementType == involvement_name::CREATED_PAVILION) {
        return std::make_unique<NewObjectsWithFtTypeCounter>(
            commitIds, coreTxn_,
            CAT_POI_URBAN,
            std::set<std::string>{FT_TYPE_URBAN_PAVILION});
    } else {
        return std::make_unique<NewObjectsCounter>(commitIds, coreTxn_, CATEGORY_PREFIX + involvementType);
    }
}

////////////////////////////////
/// COUNTERS IMPLEMENTATIONS ///
////////////////////////////////

NewObjectsCounter::NewObjectsCounter(
    std::vector<revision::DBID> commitIds,
    pqxx::transaction_base& coreTxn,
    std::string objectCategory)
    : ICommitsCounter(std::move(commitIds))
    , coreTxn_(coreTxn)
    , objectCategory_(std::move(objectCategory))
{}

int64_t NewObjectsCounter::countImpl(
    const geolib3::MultiPolygon2& polygons)
{
    rev::RevisionsGateway gateway(coreTxn_);

    auto geosMultiPolygon = polygons.polygonsNumber() ?
        geolib3::internal::geolib2geosGeometry(polygons) :
        nullptr;

    auto filter = (
        filter::Attr(objectCategory_).defined() &&
        filter::ObjRevAttr::prevCommitId().isZero() &&
        filter::ObjRevAttr::isNotDeleted() &&
        filter::ObjRevAttr::isNotRelation() &&
        filter::CommitAttr::id().in(commitIds_)
    );

    return countObjectsInsideRegion(
        gateway.reader().loadRevisions(filter),
        geosMultiPolygon
    );
}


NewObjectsLengthCounter::NewObjectsLengthCounter(
    std::vector<revision::DBID> commitIds,
    pqxx::transaction_base& coreTxn,
    std::string objectCategory)
    : ICommitsCounter(std::move(commitIds))
    , coreTxn_(coreTxn)
    , objectCategory_(std::move(objectCategory))
{}

int64_t NewObjectsLengthCounter::countImpl(
    const geolib3::MultiPolygon2& polygons)
{
    auto geosMultiPolygon = polygons.polygonsNumber() ?
        geolib3::internal::geolib2geosGeometry(polygons) :
        nullptr;

    auto filter = (
        filter::Attr(objectCategory_).defined() &&
        filter::ObjRevAttr::prevCommitId().isZero() &&
        filter::ObjRevAttr::isNotDeleted() &&
        filter::ObjRevAttr::isNotRelation() &&
        filter::CommitAttr::id().in(commitIds_) &&
        filter::Geom::defined()
    );

    rev::RevisionsGateway gateway(coreTxn_);
    auto revisions =  gateway.reader().loadRevisions(filter);

    int64_t result = 0;
    for (const auto& rev: revisions) {
        const auto& geometry = rev.data().geometry;
        ASSERT(geometry);
        const common::Geom revGeom(*geometry);
        if (geosMultiPolygon && !geosMultiPolygon->intersects(revGeom.geosGeometryPtr())) {
            continue;
        }
        double length = revGeom.realLength();
        result += llround(length);
    }
    return result;
}


ChangedRdElLengthCounter::ChangedRdElLengthCounter(
    std::vector<revision::DBID> commitIds,
    pqxx::transaction_base& coreTxn,
    std::string attribute,
    RdElChangeCountPolicy countPolicy)
    : ICommitsCounter(std::move(commitIds))
    , coreTxn_(coreTxn)
    , attribute_(std::move(attribute))
    , countPolicy_(countPolicy)
{}

int64_t ChangedRdElLengthCounter::countImpl(
    const geolib3::MultiPolygon2& polygons)
{
    RdElRevisionCounter counter = countLengthOnAnyChangeOfAttribute;
    auto filterGetter = getFilterForAnyRdElRevisions;

    switch (countPolicy_) {
        case RdElChangeCountPolicy::COUNT_ATTRIBUTE_MODIFICATION:
            counter = countLengthOnAnyChangeOfAttribute;
            filterGetter = getFilterForRdElModifyingRevisions;
            break;
        case RdElChangeCountPolicy::AVOID_CHEATING:
            counter = countLengthAvoidingCheating;
            filterGetter = getFilterForAnyRdElRevisions;
            break;
    }

    rev::RevisionsGateway gateway(coreTxn_);
    return countLengthOfChangedRdElements(
        gateway, commitIds_, polygons, filterGetter, counter, attribute_);
}


TrafficLightsCounter::TrafficLightsCounter(
    std::vector<revision::DBID> commitIds,
    pqxx::transaction_base& coreTxn)
    : ICommitsCounter(std::move(commitIds))
    , coreTxn_(coreTxn)
{}

int64_t TrafficLightsCounter::countImpl(
    const geolib3::MultiPolygon2& polygons)
{
    rev::RevisionsGateway gateway(coreTxn_);
    auto snapshot = gateway.snapshot(gateway.maxSnapshotId());

    auto geosMultiPolygon = polygons.polygonsNumber() ?
        geolib3::internal::geolib2geosGeometry(polygons) :
        nullptr;

    auto trafficLightsFilter = (
        filter::Attr(CAT_COND_TRAFFIC_LIGHT).defined() &&
        filter::ObjRevAttr::prevCommitId().isZero() &&
        filter::ObjRevAttr::isNotDeleted() &&
        filter::ObjRevAttr::isNotRelation() &&
        filter::CommitAttr::id().in(commitIds_)
    );

    auto trafficLightsRevisionIds = snapshot.revisionIdsByFilter(trafficLightsFilter);
    if (trafficLightsRevisionIds.empty()) {
        return 0;
    }

    const auto trafficLightsObjIds = [&](){
        std::vector<uint64_t> result;
        for (const auto& revId : trafficLightsRevisionIds) {
            result.push_back(revId.objectId());
        }
        return result;
    }();

    auto trafficLightsSlaveRelations = snapshot.relationsByFilter(
        filter::ObjRevAttr::masterObjectId().in(trafficLightsObjIds) &&
        filter::ObjRevAttr::isNotDeleted()
    );

    std::vector<uint64_t> slaveObjectIds;
    for (const auto& relation : trafficLightsSlaveRelations) {
        ASSERT(relation.data().relationData);
        slaveObjectIds.push_back(relation.data().relationData->slaveObjectId());
    }

    if (slaveObjectIds.empty()) {
        return 0;
    }

    auto rdJcRevisions = snapshot.objectRevisionsByFilter(
        filter::ObjRevAttr::objectId().in(slaveObjectIds) &&
        filter::ObjRevAttr::isNotRelation() &&
        filter::ObjRevAttr::isNotDeleted() &&
        filter::Attr(CAT_RD_JC).defined() &&
        filter::Geom::defined()
    );

    return countObjectsInsideRegion(
        rdJcRevisions,
        geosMultiPolygon
    );
}


CovidCommitsCounter::CovidCommitsCounter(
    std::vector<revision::DBID> commitIds,
    pqxx::transaction_base& coreTxn,
    pqxx::transaction_base& socialMasterTxn,
    const configs::editor::ConfigHolder& config)
    : ICommitsCounter(std::move(commitIds))
    , coreTxn_(coreTxn)
    , socialMasterTxn_(socialMasterTxn)
    , config_(config)
{}

int64_t CovidCommitsCounter::countImpl(
    const geolib3::MultiPolygon2& multipolygon)
{
    social::Gateway gateway(socialMasterTxn_);
    auto events = gateway.loadTrunkEditEventsByCommitIds(
        social::TIds(commitIds_.begin(), commitIds_.end())
    );

    auto eventsByNonYaUsers = filterCommitEventsByNonYaUsers(coreTxn_, events);

    int64_t resCount = 0;
    for (const auto& event : eventsByNonYaUsers) {
        const auto& category = event.getPrimaryObjectCategory();
        if (!category) {
            continue;
        }

        auto group = config_.categoryGroups().findGroupByCategoryId(category.value());

        bool countEvent =
            group &&
            COVID_CATEGORY_GROUPS.count(group->id()) &&
            isMultipolygonIntersectsBoundingBox(
                multipolygon,
                extractBoundingBoxMerc(event)
            );

        if (countEvent) {
            resCount++;
        }
    }

    return resCount;
}

} // namespace maps::wiki::tasks::involvement
