#pragma once

#include <yandex/maps/wiki/configs/editor/config_holder.h>
#include <yandex/maps/wiki/revision/objectrevision.h>
#include <maps/libs/geolib/include/multipolygon.h>

#include <maps/libs/chrono/include/time_point.h>

#include <memory>

namespace maps::wiki::tasks::involvement {

class ICounter
{
public:
    virtual ~ICounter() = default;
    virtual int64_t count(const geolib3::MultiPolygon2& polygons) = 0;
};

class ICommitsCounter : public ICounter
{
public:

    ICommitsCounter(std::vector<revision::DBID> commitIds)
        : commitIds_(std::move(commitIds))
    { }

    virtual ~ICommitsCounter() = default;

    int64_t count(const geolib3::MultiPolygon2& polygons) override
    {
        return commitIds_.empty() ? 0 : countImpl(polygons);
    }

protected:
    virtual int64_t countImpl(const geolib3::MultiPolygon2& polygons) = 0;

    const std::vector<revision::DBID> commitIds_;
};

using ICounterPtr = std::unique_ptr<ICounter>;


class CommitsCounterFactory
{
public:
    CommitsCounterFactory(
        pqxx::transaction_base& coreTxn,
        pqxx::transaction_base& socialMasterTxn,
        pqxx::transaction_base& mrcReadTxn_,
        const configs::editor::ConfigHolder& editorConfig);

    ICounterPtr createCounter(
        const std::string& involvementType,
        const std::vector<revision::DBID>& commitIds,
        std::optional<chrono::TimePoint> involvementStartOpt,
        std::optional<chrono::TimePoint> involvementFinishOpt,
        int64_t previousValue) const;

private:
    pqxx::transaction_base& coreTxn_;
    pqxx::transaction_base& socialMasterTxn_;
    pqxx::transaction_base& mrcReadTxn_;
    const configs::editor::ConfigHolder& editorConfig_;
};


////////////////////////////////
/// COUNTERS IMPLEMENTATIONS ///
////////////////////////////////


class NewObjectsCounter : public ICommitsCounter
{
public:
    NewObjectsCounter(
        std::vector<revision::DBID> commitIds,
        pqxx::transaction_base& coreTxn,
        std::string objectCategory);

private:
    int64_t countImpl(
        const geolib3::MultiPolygon2& polygons) override;

    pqxx::transaction_base& coreTxn_;
    const std::string objectCategory_;
};


class NewObjectsLengthCounter : public ICommitsCounter
{
public:
    NewObjectsLengthCounter(
        std::vector<revision::DBID> commitIds,
        pqxx::transaction_base& coreTxn,
        std::string objectCategory);

private:
    int64_t countImpl(
        const geolib3::MultiPolygon2& polygons) override;

    pqxx::transaction_base& coreTxn_;
    const std::string objectCategory_;
};


/**
 * Defines how rd_el revisions will be calculated:
 * COUNT_ATTRIBUTE_MODIFICATION policy means increasing of involvement counter
 *   on any modification of the attribute for existing rd_el.
 *   rd_el creation and deletion are not counted.
 * AVOID_CHEATING policy tries to avoid series of set-unset revisions
 *   of the attribute.
 */
enum class RdElChangeCountPolicy
{
    COUNT_ATTRIBUTE_MODIFICATION,
    AVOID_CHEATING
};


class ChangedRdElLengthCounter : public ICommitsCounter
{
public:
    ChangedRdElLengthCounter(
        std::vector<revision::DBID> commitIds,
        pqxx::transaction_base& coreTxn,
        std::string attribute,
        RdElChangeCountPolicy countPolicy);

private:
    int64_t countImpl(
        const geolib3::MultiPolygon2& polygons) override;

    pqxx::transaction_base& coreTxn_;
    const std::string attribute_;
    const RdElChangeCountPolicy countPolicy_;
};


class TrafficLightsCounter : public ICommitsCounter
{
public:
    TrafficLightsCounter(
        std::vector<revision::DBID> commitIds,
        pqxx::transaction_base& coreTxn);

private:
    int64_t countImpl(
        const geolib3::MultiPolygon2& polygons) override;

    pqxx::transaction_base& coreTxn_;
};


class CovidCommitsCounter : public ICommitsCounter
{
public:
    CovidCommitsCounter(
        std::vector<revision::DBID> commitIds,
        pqxx::transaction_base& coreTxn,
        pqxx::transaction_base& socialMasterTxn,
        const configs::editor::ConfigHolder& config);

private:
    int64_t countImpl(
        const geolib3::MultiPolygon2& polygons) override;

    pqxx::transaction_base& coreTxn_;
    pqxx::transaction_base& socialMasterTxn_;
    const configs::editor::ConfigHolder& config_;
};

class MirrorCoverageCounter : public ICounter
{
public:
    int64_t count([[maybe_unused]] const geolib3::MultiPolygon2& polygons) override
    {
        // Counter value is updated manually.
        return 0;
    }
};

} // namespace maps::wiki::tasks::involvement
