#include <maps/wikimap/mapspro/services/tasks_social/src/stats_updater/lib/stats_updater.h>

#include <maps/wikimap/mapspro/libs/sqs_client/include/aws_api.h>
#include <maps/wikimap/mapspro/libs/sqs_client/include/client.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/configs/editor/config_holder.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/tasks/include/runner.h>

namespace common = maps::wiki::common;

namespace maps::wiki::stats_updater {

namespace {

const std::string CORE_DB_ID = "long-read";
const std::string CORE_POOL_ID = "long-read";
const std::string SOCIAL_DB_ID = "social";
const std::string SOCIAL_POOL_ID = "grinder";

const std::chrono::seconds WAIT_INTERVAL{5};

} // namespace

void run(const common::ExtendedXmlDoc& config)
{
    INFO() << "starting";
    configs::editor::ConfigHolder editorConfig(
        config.get<std::string>("/config/services/editor/config"));

    maps::wiki::sqs::AwsAPI awsApi;
    sqs::Configuration sqsConfig(config);
    Aws::SQS::SQSClient sqsClient(sqs::createSqsClient(sqsConfig));

    common::PoolHolder corePoolHolder(config, CORE_DB_ID, CORE_POOL_ID);
    common::PoolHolder socialPoolHolder(config, SOCIAL_DB_ID, SOCIAL_POOL_ID);

    StatsUpdater worker(
        corePoolHolder.pool(),
        socialPoolHolder.pool(),
        editorConfig.categoryGroups(),
        sqsConfig,
        sqsClient);

    INFO() << "started";
    maps::wiki::tasks::Runner().run(
        [&] {
            worker.doWork();
        },
        WAIT_INTERVAL
    );
    INFO() << "stopped";
}

} // namespace maps::wiki::stats_updater

int main(int argc, char** argv)
try {
    maps::cmdline::Parser parser;

    auto configPath = parser.file("config")
        .help("path to services configuration");
    auto syslogTag = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    auto configDocPtr = configPath.defined()
        ? std::make_unique<common::ExtendedXmlDoc>(configPath)
        : common::loadDefaultConfig();

    maps::wiki::stats_updater::run(*configDocPtr);

    return EXIT_SUCCESS;
} catch (const maps::Exception& ex) {
    FATAL() << "Worker failed: " << ex;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Worker failed: " << ex.what();
    return EXIT_FAILURE;
}
