#include <maps/wikimap/mapspro/services/tasks_social/src/stats_updater/lib/stats_updater.h>
#include <maps/wikimap/mapspro/services/tasks_social/src/stats_updater/lib/commits_processor.h>
#include <maps/wikimap/mapspro/services/tasks_social/src/stats_updater/lib/revision_helpers.h>
#include <maps/wikimap/mapspro/libs/notifications/include/mail_event.h>

#include <yandex/maps/wiki/pubsub/commit_consumer.h>
#include <yandex/maps/wiki/social/profile_gateway.h>

#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::stats_updater {

namespace {

const size_t BATCH_SIZE_LIMIT = 1000;

notifications::MailEvent mailEventFromEditsBadge(const EditsCountBadge& badge)
{
    return notifications::MailEvent(
        badge.user,
        notifications::MailType::AchievementEditsCount,
        json::Value({{"editsCount", json::Value(badge.editsCount)}})
    );
}

} // anon namespace

StatsUpdater::StatsUpdater(
    pgpool3::Pool& corePool,
    pgpool3::Pool& socialPool,
    const configs::editor::CategoryGroups& categoryGroups,
    const sqs::Configuration& sqsConfig,
    const Aws::SQS::SQSClient& sqsClient)
        : corePool_(corePool)
        , socialPool_(socialPool)
        , categoryGroups_(categoryGroups)
        , sqsConfig_(sqsConfig)
        , sqsClient_(sqsClient)
{}

void StatsUpdater::doWork() const
{
    std::vector<pubsub::CommitId> commitIdsBatch;

    try {
        auto socialTxn = socialPool_.masterWriteableTransaction();
        pubsub::CommitConsumer consumer(*socialTxn, "StatsUpdater", revision::TRUNK_BRANCH_ID);
        consumer.setBatchSizeLimit(BATCH_SIZE_LIMIT);

        auto coreTxn = corePool_.slaveTransaction();
        commitIdsBatch = consumer.consumeBatch(*coreTxn);

        if (!commitIdsBatch.empty()) {
            INFO() << "Processing " << commitIdsBatch.size() << " commits";

            CommitsProcessor processor(*socialTxn, categoryGroups_);
            processor.processCommits(loadCommits(commitIdsBatch, *coreTxn));

            socialTxn->commit();
            socialTxn.releaseConnection();

            auto badges = processor.getEditsBadges();
            if (!badges.empty()) {
                notifications::MailEventSender sender("notifications", sqsConfig_, sqsClient_);

                auto socialReadTxn = socialPool_.slaveTransaction();
                social::ProfileGateway profileGtw(socialReadTxn.get());

                for (const auto& badge : badges) {
                    auto userProfile = profileGtw.getUserProfile(badge.user);
                    if (userProfile && !userProfile->hasBroadcastSubscription()) {
                        continue;
                    }
                    sender.send(mailEventFromEditsBadge(badge));
                }
            }
        }
    } catch (const pubsub::AlreadyLockedException&) {
    }
}

} // namespace maps::wiki::stats_updater
