#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/libs/types/include/uid.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/tasks/status_writer.h>

#include <map>
#include <boost/filesystem.hpp>

namespace pgp3 = maps::pgpool3;
namespace common = maps::wiki::common;
namespace fs = boost::filesystem;

namespace {

using maps::wiki::types::TUid;
using UidToLevelMap = std::map<TUid, int>;

const std::string NMAPS_LAUNCH_DATE = "2010-04-08";
const std::string ACTIVE_YEARS_BADGE_ID = "active_years";
constexpr size_t USERS_BATCH_SIZE = 10000;

std::string makeQueryBadges(TUid uid, int level)
{
    std::ostringstream query;
    query <<
        "INSERT INTO social.badge"
        "  (uid, badge_id, level, awarded_by)"
        "  SELECT " << uid << ", "
                    << "'" << ACTIVE_YEARS_BADGE_ID << "', "
                    << level << ", " << common::ROBOT_UID <<
        "  WHERE NOT EXISTS ("
        "   SELECT * FROM social.badge"
        "   WHERE uid = " << uid << " AND"
        "    badge_id = '" << ACTIVE_YEARS_BADGE_ID << "' AND"
        "    (level IS NULL OR level >= " << level << ")"
        "  )";
    return query.str();
}

void updateActiveYearsBadges(
    pgp3::Pool& corePgPool,
    pgp3::Pool& socialPgPool,
    const std::optional<std::string>& statusFileName)
{
    auto queryUsers =
        " SELECT uid, EXTRACT(YEAR FROM age(now(), GREATEST(created, '" + NMAPS_LAUNCH_DATE + "'))) AS level"
        " FROM acl.user"
        " WHERE status = 'active' AND NOT EXISTS ("
        "  SELECT *"
        "  FROM acl.ban_record b"
        "  WHERE b.br_uid = uid AND b.br_action = 'ban' AND b.br_expires IS NULL AND NOT EXISTS ("
        "   SELECT ub.br_id"
        "   FROM acl.ban_record ub"
        "   WHERE ub.br_uid = uid AND ub.br_action = 'unban' AND ub.br_created > b.br_created"
        "  )"
        " )"
        " ORDER BY uid";

    maps::wiki::tasks::StatusWriter statusWriter(statusFileName);
    try {
        UidToLevelMap uidToLevel;
        {
            auto coreReadTxn = corePgPool.masterReadOnlyTransaction();
            for (const auto& rowUser : coreReadTxn->exec(queryUsers)) {
                auto level = rowUser["level"].as<int>();
                if (!level) {
                    continue;
                }
                auto uid = rowUser["uid"].as<TUid>();
                uidToLevel[uid] = level;
            }
        }

        common::applyBatchOp<UidToLevelMap>(
            uidToLevel,
            USERS_BATCH_SIZE,
            [&](const UidToLevelMap& batch) {
                auto socialWriteTxn = socialPgPool.masterWriteableTransaction();

                for (const auto& [uid, level] : batch) {
                    socialWriteTxn->exec(makeQueryBadges(uid, level));
                }
                socialWriteTxn->commit();
            });

        statusWriter.flush();
    } catch (const std::exception& e) {
        ERROR() << "Task failed: " << e.what();
        statusWriter.err(e.what());
        statusWriter.flush();
        throw;
    }
}

} // namespace

int main(int argc, char* argv[]) try
{
    maps::cmdline::Parser parser;
    auto workerConfig = parser.string("config")
            .help("path to worker configuration");
    auto syslogTag = parser.string("syslog-tag")
            .help("redirect log output to syslog with given tag");
    auto statusDir = parser.string("status-dir")
            .help("path to status dir");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    std::unique_ptr<common::ExtendedXmlDoc> configDocPtr;
    if (workerConfig.defined()) {
        configDocPtr.reset(new common::ExtendedXmlDoc(workerConfig));
    } else {
        configDocPtr = common::loadDefaultConfig();
    }

    common::PoolHolder socialDbHolder(*configDocPtr, "social", "grinder");
    auto& socialPgPool = socialDbHolder.pool();

    maps::pgp3utils::PgAdvisoryXactMutex locker(
        socialPgPool,
        static_cast<int64_t>(common::AdvisoryLockIds::UPDATE_BADGES));
    if (!locker.try_lock()) {
        INFO() << "Database is already locked. Task interrupted.";
        return EXIT_SUCCESS;
    }

    common::PoolHolder coreDbHolder(*configDocPtr, "core", "grinder");
    auto& corePgPool = coreDbHolder.pool();

    std::optional<std::string> statusFileName;
    if (statusDir.defined()) {
        fs::path filePath(statusDir);
        filePath /= "wiki-update-badges-worker.status";
        statusFileName = filePath.string();
    }

    INFO() << "Started updating user badges";
    updateActiveYearsBadges(corePgPool, socialPgPool, statusFileName);

    INFO() << "Finished updating user badges";
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << "Worker failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
