#include "component_feed_cache.h"

#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/diff.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/retry_duration.h>

using namespace std::string_literals;

namespace maps::wiki::poi {
namespace {
const auto WORKER_COMMIT_SEARCH_CACHE_TABLE = "sprav.export_poi_worker_component_feed_cache"s;

bool isSpravRobot(UserID uid)
{
    return uid == common::WIKIMAPS_SPRAV_UID ||
        uid == common::WIKIMAPS_SPRAV_COMMON_UID;
}

std::set<DBID>
commitIdsFromRevisions(const RevisionIds& revisionIds)
{
    std::set<DBID>  commitIds;
    for (const auto revisionId : revisionIds) {
        commitIds.insert(revisionId.commitId());
    }
    return commitIds;
}

} // namespace

ComponentFeedData::ComponentFeedData(const pqxx::row& row)
    : stored_(true)
    , objectId_(row["object_id"].as<DBID>())
    , recentCommitId_(row["recent_commit_id"].as<DBID>())
{
    const auto componentsJson = maps::json::Value::fromString(row["components_json_data"].as<std::string>());
    for (const auto& field : componentsJson.fields()) {
        for (auto* component : components()) {
            if (component->name() == field) {
                component->readJson(componentsJson[field]);
            }
        }
    }
}

ComponentFeedData::ComponentFeedData(DBID objectId, DBID recentCommitId)
    : stored_(false)
    , objectId_(objectId)
    , recentCommitId_(recentCommitId)
{
}

std::vector<FeedComponentBase*>
ComponentFeedData::components()
{
    std::vector<FeedComponentBase*> result;
    result.push_back(&lonLatComponent_);
    result.push_back(&namesComponent_);
    return result;
}

std::vector<const FeedComponentBase*>
ComponentFeedData::components() const
{
    std::vector<const FeedComponentBase*> result;
    result.push_back(&lonLatComponent_);
    result.push_back(&namesComponent_);
    return result;
}

std::string
ComponentFeedData::json() const
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder object) {
        for (const auto* component : components()) {
            object[component->name()] = [&](maps::json::ObjectBuilder componentBuilder) {
                component->updateJson(componentBuilder);
            };
        }
    };
    return builder.str();
}

void
ComponentFeedData::markToStoreWithCommit(DBID commitId)
{
    ASSERT(commitId >= recentCommitId_);
    recentCommitId_ = commitId;
    stored_ = false;
}

ComponentFeedCache::ComponentFeedCache(DBIDSet oids, const Config& cfg)
    : cfg_(cfg)
    , cacheHits_(0)
    , cacheMiss_(0)
{
    if (oids.empty()) {
        return;
    }
    const auto rows = common::retryDuration([&] {
        auto socialTxn = cfg.socialPool().slaveTransaction();
        return socialTxn->exec(
            "SELECT * FROM " + WORKER_COMMIT_SEARCH_CACHE_TABLE +
            " WHERE " + maps::wiki::common::whereClause("object_id", oids));
    });
    for (const auto& row : rows) {
        cache_.emplace(row["object_id"].as<DBID>(), ComponentFeedData(row));
    }
}

ComponentFeedCache::~ComponentFeedCache()
{
    INFO() << "Component cache stat. Misses: "
        << cacheMiss_ << " Hits: " << cacheHits_;
}

void
ComponentFeedCache::saveModified()
{
    common::retryDuration([&] {
        auto socialTxn = cfg_.socialPool().masterWriteableTransaction();
        for (const auto& [id, data] : cache_) {
            if (data.stored()) {
                continue;
            }
            socialTxn->exec("DELETE FROM " + WORKER_COMMIT_SEARCH_CACHE_TABLE +
                " WHERE object_id = " + std::to_string(id));
            socialTxn->exec("INSERT INTO " + WORKER_COMMIT_SEARCH_CACHE_TABLE +
                " (object_id, recent_commit_id, components_json_data) "
                " VALUES(" + std::to_string(data.objectId()) +
                ", " + std::to_string(data.recentCommitId()) +
                ", " + socialTxn->quote(data.json()) + ")");
        }
        socialTxn->commit();
    });
}

DBID
ComponentFeedCache::resetCache(DBID objectId)
{
    auto it = cache_.find(objectId);
    if (it == cache_.end()) {
        return 0;
    }
    auto recentCommitId = it->second.recentCommitId();
    cache_.erase(objectId);
    return recentCommitId;
}

const ComponentFeedData&
ComponentFeedCache::get(DBID objectId, DBID recentCommitId)
{
    DEBUG() << "Component get objectId:" << objectId << " recentCommitId:" << recentCommitId;
    auto it = cache_.find(objectId);
    if (it != cache_.end() && it->second.recentCommitId() >= recentCommitId) {
        ++cacheHits_;
        return it->second;
    }
    ++cacheMiss_;
    if (it == cache_.end()) {
        cache_.emplace(objectId, ComponentFeedData(objectId, recentCommitId));
    }
    ComponentFeedData& objectCommitsData = cache_.at(objectId);
    DBID lowestCommit =
        it == cache_.end() ? 0 : objectCommitsData.recentCommitId(); 

    const auto revsFilter =
        revision::filters::ObjRevAttr::objectId() == objectId &&
        revision::filters::ObjRevAttr::isNotRelation() &&
        revision::filters::ObjRevAttr::commitId() > lowestCommit &&
        revision::filters::ObjRevAttr::commitId() <= recentCommitId;

    const auto slaveRelsRevsFilter =
        revision::filters::ObjRevAttr::masterObjectId() == objectId &&
        revision::filters::ObjRevAttr::isRelation() &&
        revision::filters::ObjRevAttr::commitId() > lowestCommit &&
        revision::filters::ObjRevAttr::commitId() <= recentCommitId;
    
    common::retryDuration([&] {
        auto coreTxn = cfg_.mainPool().slaveTransaction();
        revision::RevisionsGateway gateway(*coreTxn, cfg_.branch());
        auto snapshot = gateway.historicalSnapshot(recentCommitId);
        const auto revisionIds = snapshot.revisionIdsByFilter(revsFilter);
        const auto slaveRelsRevsIds = snapshot.revisionIdsByFilter(slaveRelsRevsFilter);
        auto commitIds = commitIdsFromRevisions(revisionIds);
        auto relsCommitIds = commitIdsFromRevisions(slaveRelsRevsIds);
        commitIds.insert(relsCommitIds.begin(), relsCommitIds.end());
        const auto commits = loadVisibleApprovedUserCommits(*coreTxn, commitIds);
        auto feedComponents = objectCommitsData.components();
        for (const auto* commit : commits) {
            DEBUG() << "Proccessing commit:" << commit->id();
            feedComponents.erase(
                std::remove_if(
                    feedComponents.begin(),
                    feedComponents.end(),
                    [&](auto* feedComponent) {
                        return feedComponent->updateFromCommit(cfg_.branch(), *coreTxn, *commit, objectId);
                    }),
                feedComponents.end());
            if (feedComponents.empty()) {
                break;
            }
        }
    });
    objectCommitsData.markToStoreWithCommit(recentCommitId);
    return objectCommitsData;
}

std::vector<const revision::Commit*>
ComponentFeedCache::loadVisibleApprovedUserCommits(pqxx::transaction_base& coreTxn, const std::set<DBID>& ids)
{
    std::vector<DBID> notCached;
    for (const auto& commitId : ids) {
        if (!commitsCache_.count(commitId)) {
            notCached.push_back(commitId);
        }
    }
    if (!notCached.empty()) {
        auto commitFilter =
            revision::filters::CommitAttr::isVisible(cfg_.branch()) &&
            revision::filters::CommitAttr::id().in(notCached);
        auto commits = revision::Commit::load(coreTxn, commitFilter);
        for (const auto& commit : commits) {
            if (commit.state() == revision::CommitState::Approved) {
                commitsCache_.emplace(commit.id(), commit);
            }
        }
    }
    std::vector<const revision::Commit*> commits;
    for (const auto& commitId : ids) {
        const auto it = commitsCache_.find(commitId);
        if (it != commitsCache_.end() &&
            !isSpravRobot(it->second.createdBy()))
        {
            commits.push_back(&it->second);
        }
    }
    std::sort(commits.begin(), commits.end(),
        [](const auto* lc, const auto*rc) {
            return lc->id() > rc->id();
        });
    return commits;
}
} // namespace maps::wiki::poi
