#include "config.h"

#include "utils.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/tasks/export.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/xml/include/xml.h>
#include <yandex/maps/wiki/common/retry_duration.h>

using namespace std::chrono_literals;

namespace maps::wiki::poi {
namespace {

const std::string APPROVED_BRANCH = "approved";
const std::string RELEASE_BRANCH = "release";

std::unique_ptr<pgpool3::Pool> makePool(const std::string& connStr)
{
    pgpool3::PoolConstants poolConstants(1, 2, 0, 0);
    poolConstants.getTimeoutMs = 20s;
    poolConstants.pingIntervalMs = 5s;
    poolConstants.pingTimeoutMs = 60s;

    return std::make_unique<pgpool3::Pool>(
        connStr,
        std::move(poolConstants));
}

} // anonymous namespace

Config::Config(const common::ExtendedXmlDoc& configXml,
               const std::string& branchStr,
               chrono::TimePoint timePoint,
               const std::string& tempDirPath,
               const std::string& mappingContent,
               const std::string& yMapsDfXmlPath)
    : configXml_(configXml)
    , mainPool_(configXml_, "long-read", "long-read")
    , socialPool_(configXml_, "social", "grinder")
    , branch_(loadBranch(branchStr))
    , snapshotId_(maxSnapshotId())
    , yMapsDfXmlPath_(yMapsDfXmlPath)
    , mdsConfig_(makeMdsConfig())
    , resourceName_(makeResourceName(
          snapshotId_.hasApproveOrder() ? snapshotId_.approveOrder() : 0,
          timePoint))
    , tempDirPath_(tempDirPath)
    , taskDirPath_(tempDirPath_ + "/" + resourceName_)
    , jsonDirPath_(taskDirPath_ + "/json")
    , resultFilePathCalculatedRubricId_(taskDirPath_ + "/poi_export_calculated_rubric_id.xml.gz")
    , resultFilePathAccurateRubricId_(taskDirPath_ + "/poi_export_accurate_rubric_id.xml.gz")
    , md5FilePathCalculatedRubricId_(resultFilePathCalculatedRubricId_ + ".md5")
    , md5FilePathAccurateRubricId_(resultFilePathAccurateRubricId_ + ".md5")
    , resultFilePathVerifiedCoordsFeed_(taskDirPath_ + "/poi_export_verified_coords_feed.xml.gz")
    , md5FilePathVerifiedCoordsFeed_(resultFilePathVerifiedCoordsFeed_ + ".md5")
    , resultFilePathNamesFeed_(taskDirPath_ + "/poi_export_names_feed.xml.gz")
    , md5FilePathNamesFeed_(resultFilePathNamesFeed_ + ".md5")
    , resultFilePathCoordsFeed_(taskDirPath_ + "/poi_export_coords_feed.xml.gz")
    , md5FilePathCoordsFeed_(resultFilePathCoordsFeed_ + ".md5")
    , resultFilePathIndoorFeed_(taskDirPath_ + "/poi_export_indoor_feed.xml.gz")
    , md5FilePathIndoorFeed_(resultFilePathIndoorFeed_ + ".md5")
    , resultFilePathIndoorNamesFeed_(taskDirPath_ + "/poi_export_indoor_names_feed.xml.gz")
    , md5FilePathIndoorNamesFeed_(resultFilePathIndoorNamesFeed_ + ".md5")
    , resultFilePathEntrancesFeed_(taskDirPath_ + "/poi_export_entrances_feed.yson.gz")
    , md5FilePathEntrancesFeed_(resultFilePathEntrancesFeed_ + ".md5")
    , printedConfig_(taskDirPath_ + "/printed-config.json")
{
    xml3::Doc doc(mappingContent, xml3::Doc::Source::String);
    auto nodes = doc.nodes("/rubrics_mapping/pair");
    for (size_t i = 0, count = nodes.size(); i < count; ++i) {
        auto pair = nodes[i];
        bool byParent = pair.attr<bool>("by_parent", false);
        if (byParent) {
            parentMapping_.emplace(
                pair.attr<FtTypeId>("ft_type_id"),
                pair.attr<RubricId>("rubric_id"));
        } else {
            mapping_.emplace(
                pair.attr<FtTypeId>("ft_type_id"),
                pair.attr<RubricId>("rubric_id"));
        }
        INFO() << " Mapping ft_type_id: " << pair.attr<FtTypeId>("ft_type_id") << " to rubric id: " << pair.attr<RubricId>("rubric_id");
    }
}

Config::Config(const common::ExtendedXmlDoc& configXml,
               const std::string& mappingContent,
               const std::string& yMapsDfXmlPath)
    : Config(configXml,
             RELEASE_BRANCH,
             chrono::TimePoint::clock::now(),
             TEMP_DIR_DEFAULT_PATH,
             mappingContent,
             yMapsDfXmlPath)
{
}

revision::Branch Config::loadBranch(const std::string& branchStr) const
{
    return common::retryDuration([&] {
        auto txn = mainPool().slaveTransaction();
        revision::BranchManager branchManager(txn.get());
        if (branchStr == RELEASE_BRANCH) {
            auto branches = branchManager.load(
                {
                    { revision::BranchType::Stable, 1}
                });
            if (!branches.empty() && branches.front().state() != revision::BranchState::Unavailable ) {
                INFO() << "Using branch: " << branches.front().id();
                return branches.front();
            }
            WARN() << "No suitable stable branch found, fallback to approved.";
            return branchManager.loadByString(APPROVED_BRANCH);
        }
        return branchManager.loadByString(branchStr);
    });
}

revision::SnapshotId Config::maxSnapshotId() const
{
    return common::retryDuration([&] {
        auto txn = mainPool().slaveTransaction();
        revision::RevisionsGateway gateway(txn.get(), branch());
        return gateway.maxSnapshotId();
    });
}

mds::Configuration Config::makeMdsConfig() const
{
    auto mdsXpath = tasks::getConfigExportXpath() + "/mds";
    auto getMdsAttr = [&] (const std::string& name) {
        return configXml_.getAttr<std::string>(mdsXpath, name);
    };

    mds::Configuration mdsConfig(getMdsAttr("host"),
                                 getMdsAttr("namespace-name"),
                                 getMdsAttr("auth-header"));
    mdsConfig.setPathPrefix("export")
        .setMaxRequestAttempts(10)
        .setRetryInitialTimeout(std::chrono::seconds(1))
        .setRetryTimeoutBackoff(2);
    return mdsConfig;
}

std::string Config::yMapsDfConnStr() const
{
    return connStr_.empty()
        ? tasks::defaultExportDatabase(configXml_)
        : connStr_;
}

pgpool3::Pool& Config::yMapsDfPool() const
{
    if (!yMapsDfPool_) {
        yMapsDfPool_ = makePool(yMapsDfConnStr());
    }
    return *yMapsDfPool_;
}

} // maps::wiki::poi
