#include "dump.h"

#include "formatter.h"
#include "names_feed_formatter.h"
#include "coords_feed_formatter.h"
#include "indoor_formatter.h"
#include "indoor_names_formatter.h"
#include "entrances_formatter.h"

#include "loader.h"
#include "poi_def.h"
#include "search_path_guard.h"
#include "component_feed/component_feed_cache.h"

#include <maps/wikimap/mapspro/libs/poi_feed/include/feed_object_data.h>
#include <maps/libs/common/include/make_batches.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_stream.hpp>

#include <filesystem>

namespace fs = std::filesystem;
namespace io = boost::iostreams;
using namespace std::string_literals;

namespace maps {
namespace wiki {
namespace poi {
namespace {

constexpr size_t BATCH_SIZE = 1000;
const auto WORKER_FEED_DATA_TABLE = "sprav.export_poi_worker_feed_data"s;

class GZipWriter {
public:
    GZipWriter(const fs::path& path) : out_(path)
    {
        gzippedOut_.push(io::gzip_compressor());
        gzippedOut_.push(out_);
    }

    std::ostream& stream() { return gzippedOut_; }

private:
    std::ofstream out_;
    io::filtering_ostream gzippedOut_;
};

poi_feed::FeedObjectData
createFeedObjectData(const PoiDef& def)
{
    ASSERT(def.id && def.actualizationDate);
    poi_feed::FeedObjectData data;
    data.setNmapsId(*def.id);
    if (!def.permalink.empty() && def.permalinkFromReserves == PermalinkFromReserves::False) {
        data.setPermalink(boost::lexical_cast<poi_feed::PermalinkId>(def.permalink));
    }
    data.setActualizationDate(*def.actualizationDate);
    if (def.lon && def.lat) {
        data.setPosition(
            poi_feed::FeedObjectData::Position {
                *def.lon,
                *def.lat
            });
    }
    data.setNames(def.names);
    data.setShortNames(def.shortNames);
    data.setRubricId(def.userDefinedRubricId);
    for (const auto rubricId : def.userDefinedSecondaryRubricIds) {
        data.addSecondaryRubric(rubricId);
    }
    if (def.ftTypeId) {
        data.setFtTypeId(*def.ftTypeId);
    }
    data.setIndoorLevelUniversal(def.indoorLevelUniversal);
    if (def.indoorPlanId) {
        data.setIndoorPlanId(*def.indoorPlanId);
    }
    if (def.isGeoproduct) {
        data.setIsGeoproduct(*def.isGeoproduct);
    }
    return data;
}

ObjectIdToPermalinkUnknownMap
loadNyakMappingUnknownPermalinks(const Config& cfg)
{
    auto rows = common::retryDuration([&] {
        auto txn = cfg.socialPool().slaveTransaction();
        return txn->exec("SELECT object_id, permalink FROM sprav.merge_poi_altay_feed_unknown;");
    });
    ObjectIdToPermalinkUnknownMap data;
    data.reserve(rows.size());
    for (const auto row : rows) {
        data.emplace(
            row["object_id"].as<DBID>(),
            row["permalink"].as<DBID>()
        );
    }
    return data;
}

void cleanExportFeedDatabase(const Config& cfg)
{
    common::retryDuration([&] {
        auto txn = cfg.socialPool().masterWriteableTransaction();
        txn->exec("TRUNCATE " + WORKER_FEED_DATA_TABLE);
        txn->commit();
    });
}

void writeToExportFeedDatabase(const poi_feed::FeedObjectDataVector& exportFeedBatch, const Config& cfg)
{
    common::retryDuration([&] {
        auto txn = cfg.socialPool().masterWriteableTransaction();
        std::stringstream query;
        query <<
            "INSERT INTO " + WORKER_FEED_DATA_TABLE
            + " (object_id, revision, data_json) VALUES ";
        bool first = true;
        for (const auto& data : exportFeedBatch) {
            if (!first) {
                query << ",";
            }
            first = false;
            query
                << "("
                << data.nmapsId() << ", "
                << chrono::sinceEpoch<std::chrono::milliseconds>(data.actualizationDate()) << ", "
                << txn->quote(data.toJson())
                << ")";
        }
        if (first) {
            return;
        }
        txn->exec(query.str());
        txn->commit();
    });
}

template<typename FormatterType>
class FeedWriterStreamHelper
{
public:
    explicit FeedWriterStreamHelper(const fs::path& path)
        : gzipWriter_(path)
        , out_(gzipWriter_.stream())
        , formatter_(out_)
    {
    }

    ~FeedWriterStreamHelper()
    {
        out_.flush();
    }
    FormatterType* operator ->() { return &formatter_; }

private:
    GZipWriter gzipWriter_;
    std::ostream& out_;
    FormatterType formatter_;
};

} // namespace

void dump(const Config& cfg)
{
    auto txnYMapsDf = cfg.yMapsDfPool().masterReadOnlyTransaction();
    auto txnTds = cfg.mainPool().masterReadOnlyTransaction();
    revision::RevisionsGateway gateway(txnTds.get(), cfg.branch());
    auto snapshot = gateway.snapshot(cfg.snapshotId());
    SearchPathGuard searchPathGuard{txnYMapsDf.get(), cfg.resourceName()};

    FeedWriterStreamHelper<XmlFormatter> formatter(cfg.resultFilePathCalculatedRubricId());
    FeedWriterStreamHelper<XmlFormatter> verifiedCoordsFeedFormatter(cfg.resultFilePathVerifiedCoordsFeed());
    FeedWriterStreamHelper<XmlFormatter> formatterAccurateRubric(cfg.resultFilePathAccurateRubricId());
    FeedWriterStreamHelper<NamesXmlFormatter> namesFeedFormatter(cfg.resultFilePathNamesFeed());
    FeedWriterStreamHelper<CoordsXmlFormatter> coordsFeedFormatter(cfg.resultFilePathCoordsFeed());
    FeedWriterStreamHelper<IndoorXmlFormatter> indoorFeedFormatter(cfg.resultFilePathIndoorFeed());
    FeedWriterStreamHelper<IndoorNamesXmlFormatter> indoorNamesFeedFormatter(cfg.resultFilePathIndoorNamesFeed());
    FeedWriterStreamHelper<EntrancesYsonFormatter> entrancesFeedFormatter(cfg.resultFilePathEntrancesFeed());

    Loader loader(
        txnYMapsDf.get(),
        cfg.socialPool(),
        snapshot,
        cfg.mapping(),
        cfg.parentMapping(),
        loadNyakMappingUnknownPermalinks(cfg));
    const auto poiIds = loader.loadPoiIds();
    INFO() << "loaded POI IDs: " << poiIds.size();

    const auto batches = maps::common::makeBatches(poiIds, BATCH_SIZE);
    cleanExportFeedDatabase(cfg);

    for (size_t i = 0; i < batches.size(); ++i) {
        INFO() << "dump pass " << i + 1 << "/" << batches.size();
        const auto& batch = batches[i];
        auto poiDefs = loader.loadPoiDefs({batch.begin(), batch.end()});

        poi_feed::FeedObjectDataVector exportFeed;
        exportFeed.reserve(poiDefs.size());

        DBIDSet feedCacheIds;
        DBIDSet indoorFeedCacheIds;

        std::vector<const PoiDef*> componentFeedPois;
        componentFeedPois.reserve(poiDefs.size());

        std::vector<const PoiDef*> indoorComponentFeedPois;
        indoorComponentFeedPois.reserve(poiDefs.size());

        for (auto& poiDef : poiDefs) {
            if (!poiDef.indoorLevelUniversal.empty()) {
                if (!poiDef.permalink.empty()) {
                    indoorFeedFormatter->append(poiDef);
                    if (isFullMergeImplemented(poiDef)) {
                        indoorComponentFeedPois.emplace_back(&poiDef);
                        indoorFeedCacheIds.insert(*poiDef.id);
                    }
                }
            } else {
                if (poiDef.userDefinedRubricId) {
                    formatterAccurateRubric->append(poiDef);
                } else {
                    formatter->append(poiDef);
                }
                if (poiDef.hasVerifiedCoords == HasVerifiedCoords::True) {
                    verifiedCoordsFeedFormatter->append(poiDef);
                }
                if (isFullMergeImplemented(poiDef) && !poiDef.permalink.empty()) {
                    componentFeedPois.emplace_back(&poiDef);
                    feedCacheIds.insert(*poiDef.id);
                }
            }
            if (poiDef.entrances) {
                entrancesFeedFormatter->append(poiDef);
            }
            exportFeed.push_back(createFeedObjectData(poiDef));
        }

        ComponentFeedCache feedCache(feedCacheIds, cfg);
        for (const auto* poiDef : componentFeedPois) {
            const auto& objectFeedData = feedCache.get(*poiDef->id, *poiDef->recentCommitId);
            const auto& names = objectFeedData.names();
            if (names.hasData()) {
                namesFeedFormatter->append(*poiDef, names);
            }
            const auto& coords = objectFeedData.coords();
            if (coords.hasData()) {
                coordsFeedFormatter->append(*poiDef, coords);
            }
        }

        ComponentFeedCache indoorFeedCache(indoorFeedCacheIds, cfg);
        for (const auto* poiDef : indoorComponentFeedPois) {
            const auto& objectFeedData = indoorFeedCache.get(*poiDef->id, *poiDef->recentCommitId);
            const auto& names = objectFeedData.names();
            if (names.hasData()) {
                indoorNamesFeedFormatter->append(*poiDef, names);
            }
        }

        feedCache.saveModified();
        writeToExportFeedDatabase(exportFeed, cfg);
    }
    INFO() << "POIs are dumped";
}

} // namespace poi
} // namespace wiki
} // namespace maps
