#include "loader_helpers.h"

#include <yandex/maps/wiki/revision/commit.h>
#include <maps/libs/log8/include/log8.h>
#include <boost/lexical_cast.hpp>

namespace maps::wiki::poi {

namespace {
const auto PROPKEY_PRIMARY_OBJECT_PREFIX = "primary_object:"s;

void
updateObjectMaxCommit(
    DBID poiId,
    DBID commitId,
    ObjectIdToCommitIdMap& objectIdToMaxCommitId)
{
    auto mapResult = objectIdToMaxCommitId.emplace(poiId, commitId);
    if (!mapResult.second) {
        mapResult.first->second = std::max(mapResult.first->second, commitId);
    }
}

DBID
commitPrimaryObjectId(const revision::Commit& commit)
{
    const auto& attrs = commit.attributes();
    for (const auto& [name, value] : attrs) {
        if (name.starts_with(PROPKEY_PRIMARY_OBJECT_PREFIX)) {
            return boost::lexical_cast<DBID>(name.substr(
                PROPKEY_PRIMARY_OBJECT_PREFIX.length()));
        }
    }
    return 0;
}

} // namespace

void updateMapWithRecentSlaveRevisions(
    const DBIDSet& poiIds,
    const Snapshot& tds,
    ObjectIdToCommitIdMap& objectIdToMaxCommitId)
{
    const auto slaveRelationsRevIds = tds.revisionIdsByFilter(
        revision::filters::ObjRevAttr::isRelation()
        && revision::filters::ObjRevAttr::masterObjectId().in(poiIds));

    DBIDSet knownCommits;
    for (const auto& [objId, commitId] : objectIdToMaxCommitId) {
        knownCommits.insert(commitId);
    }
    DBIDSet commitIds;
    for (const auto& revId : slaveRelationsRevIds) {
        if (!knownCommits.count(revId.commitId())) {
            commitIds.insert(revId.commitId());
        }
    }
    if (commitIds.empty()) {
        return;
    }
    auto commitFilter = revision::filters::CommitAttr::id().in(commitIds);
    auto commits = revision::Commit::load(tds.reader().work(), commitFilter);
    DEBUG() << "Relations change commits loaded:" << commits.size();
    DBIDSet notFoundByNotesCommitIds;
    for (const auto& commit : commits) {
        auto primaryObjectId = commitPrimaryObjectId(commit);
        if (!primaryObjectId || !poiIds.count(primaryObjectId)) {
            DEBUG() << "No primaryObjectId in commit:" << commit.id();
            notFoundByNotesCommitIds.insert(commit.id());
        } else {
            updateObjectMaxCommit(primaryObjectId, commit.id(), objectIdToMaxCommitId);
        }
    }
    std::vector<RevisionID> revisionsInGroupCommits;
    for (const auto& revId : slaveRelationsRevIds) {
        if (notFoundByNotesCommitIds.count(revId.commitId())) {
            revisionsInGroupCommits.push_back(revId);
        }
    }
    const auto relDatas = tds.reader().loadRevisions(revisionsInGroupCommits);
    DEBUG() << "Relations change loaded:" << relDatas.size();
    for (const auto& relData : relDatas) {
        ASSERT(relData.data().relationData);
        const auto masterId = relData.data().relationData->masterObjectId();
        if (!poiIds.count(masterId)) {
            continue;
        }
        DEBUG() << "Found group relation commit:" << relData.id().commitId() << " for object:" << masterId;
        updateObjectMaxCommit(masterId, relData.id().commitId(), objectIdToMaxCommitId);
    }
}
} // namespace maps::wiki::poi
