#include "revision2json.h"

#include "json2ymapsdf.h"

#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revisionapi/revisionapi.h>
#include <yandex/maps/wiki/threadutils/executor.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <boost/format.hpp>

#include <filesystem>
#include <fstream>
#include <iomanip>

namespace fs = std::filesystem;

namespace maps {
namespace wiki {
namespace poi {
namespace {

const size_t BATCH_LIMIT = 10000;
const size_t TASK_LIMIT = 1000000;
const std::string FILE_NAME_SUFFIX = "-%|03d|.json";

revision::DBID lastObjectId(pgpool3::Pool& pool)
{
    return common::retryDuration([&] {
        auto txn = pool.masterReadOnlyTransaction();
        revision::RevisionsGateway gtw(*txn);
        return gtw.lastObjectId();
    });
}

void ensureDirectory(const fs::path& path)
{
    if (fs::exists(path)) {
        REQUIRE(fs::is_directory(path),
                "output dir path '" + path.string()
                    + "' exists and is not a directory");
    }
    else {
        fs::create_directories(path);
    }
}

void exportJsons(
    const Config& cfg,
    const revisionapi::ExportParams& params,
    const revisionapi::GetStreamForChunkFunc& callback,
    const revisionapi::FilterPtr& filter)
{
    std::map<size_t, std::shared_ptr<std::stringstream>> streams;

    auto memoryWriter = [&](size_t index) -> std::shared_ptr<std::ostream> {
        auto newStream = std::make_shared<std::stringstream>();
        streams[index] = newStream;
        return newStream;
    };

    auto& pool = cfg.mainPool();

    common::retryDuration([&] {
        streams.clear();
        try {
            revisionapi::RevisionAPI api(pool);
            api.exportData(params, memoryWriter, filter);
        } catch (const revisionapi::DataError& ex) {
            throw common::RetryDurationCancel() << ex.what();
        }
    }); 

    for (auto& [index, streamPtr] : streams) {
        auto newStreamPtr = callback(index);
        *newStreamPtr << streamPtr->rdbuf();
    }
}

} // namespace

void revision2json(
    const Config& cfg, 
    const std::string& json2ymapsdfPath)
{
    fs::path jsonDirPath = fs::absolute(fs::path(cfg.jsonDirPath()));
    ensureDirectory(jsonDirPath);

    json2ymapsdf(cfg, Json2YMapsDfMode::PrintConfig, json2ymapsdfPath);
    auto params = revisionapi::ExportParams::loadFromFile(
        cfg.branch(), cfg.snapshotId().commitId(), cfg.printedConfig());
    params->setWriteBatchSize(BATCH_LIMIT);
    params->setEmptyJsonPolicy(revisionapi::EmptyJsonPolicy::Skip);

    std::vector<ThreadPool::Functor> tasks;
    revision::DBID id = 0;
    const auto lastId = lastObjectId(cfg.mainPool());
    while (id <= lastId) {
        const auto endId = std::min(lastId, id + TASK_LIMIT - 1);
        tasks.emplace_back([&cfg, &params, jsonDirPath, id, endId, lastId] {
            INFO() << "json pass " << id / TASK_LIMIT + 1 << "/"
                   << lastId / TASK_LIMIT + 1;

            revisionapi::GetStreamForChunkFunc callback =
                [jsonDirPath, id](size_t chunkNo) {
                    std::ostringstream fileNamePattern;
                    fileNamePattern << std::setw(8) << std::setfill('0')
                                    << id / TASK_LIMIT << FILE_NAME_SUFFIX;

                    auto path = jsonDirPath
                                / (boost::format(fileNamePattern.str())
                                   % chunkNo).str();
                    std::unique_ptr<std::ostream> stream(
                        new std::ofstream(path.string()));
                    REQUIRE(!stream->fail(), "Error opening '" + path.string()
                                                 + "' for writing");
                    return stream;
                };
            revision::DBIDSet objectIdBetweenFilter = {id, endId};
            revisionapi::FilterPtr filter
                = std::make_shared<revision::filters::TableAttrFilterExpr>(
                    revision::filters::ObjRevAttr::objectId().between(
                        objectIdBetweenFilter));

            exportJsons(cfg, *params, callback, filter);
        });

        id = endId + 1;
    }

    Executor executor;
    for (size_t i = 0; i < tasks.size(); ++i) {
        executor.addTask([&tasks, i] { tasks[i](); });
    }

    const auto slaveMaxSize = cfg.mainPool().state().constants.slaveMaxSize;
    INFO() << "Working threads size: " << slaveMaxSize;
    ThreadPool threadPool(slaveMaxSize);
    executor.executeAllInThreads(threadPool);
}

} // poi
} // wiki
} // maps
