#include <maps/wikimap/mapspro/services/tasks_sprav/src/export_poi_worker/formatter.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/export_poi_worker/entrances_formatter.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/export_poi_worker/poi_def.h>

#include <boost/test/unit_test.hpp>

#include <sstream>
#include <vector>
#include <string>
#include <ctime>

using namespace maps::wiki::poi;

namespace {

void checkContainsInOrder(const std::string& text,
                          const std::vector<std::string>& items)
{
    std::size_t pos = 0;
    for (const auto& item : items) {
        auto where = text.find(item, pos);
        BOOST_CHECK_MESSAGE(where != text.npos,
                            "String " << text << " doesn't contain substring "
                                      << item << " after position " << pos);
        if (where != text.npos) {
            pos = where + item.size();
        }
    }
}

} // anonymous namespace

BOOST_AUTO_TEST_CASE(simple_xml_formatter_test)
{
    std::ostringstream out;
    PoiDef def;
    def.id = 100500;
    def.userDefinedRubricId = 40;
    def.userDefinedSecondaryRubricIds.insert(41);
    def.userDefinedSecondaryRubricIds.insert(42);
    def.actualizationDate = maps::chrono::TimePoint::clock::now();
    def.lon = 72.2;
    def.lat = 74.8;
    def.dispClass = 0;
    def.names = {{"en", "english name"}, {"ru", "russian name"}};
    def.shortNames
        = {{"en", "english shortname"}, {"ru", "russian shortname"}};
    def.phones = {"(555) 555-55-55", "(666) 666-66-66"};
    def.permalink = "123456789";
    def.entrances = {{100501, 72.1, 74.7}, {100502, 72.3, 74.9}};
    def.indoorPlanId = 13787134433;
    {
        XmlFormatter formatter(out);
        formatter.append(def);
    }
    checkContainsInOrder(
        out.str(), {"<?xml",
                    "UTF-8",
                    "<companies",
                    "<company>",
                    "<company-id>100500</company-id>",
                    "<name lang=\"en\">english name</name>",
                    "<name lang=\"ru\">russian name</name>",
                    "<shortname lang=\"en\">english shortname</shortname>",
                    "<shortname lang=\"ru\">russian shortname</shortname>",
                    "<coordinates>",
                    "<lon>72.2",
                    "</lon>",
                    "<lat>74.8",
                    "</lat>",
                    "</coordinates>",
                    "<phone>",
                    "<number>(555) 555-55-55</number>",
                    "</phone>",
                    "<phone>",
                    "<number>(666) 666-66-66</number>",
                    "</phone>",
                    "<rubric-id>40</rubric-id>",
                    "<secondary-rubric-id>41</secondary-rubric-id>",
                    "<secondary-rubric-id>42</secondary-rubric-id>",
                    "<actualization-date>",
                    "</actualization-date>",
                    "<permalink>123456789</permalink>",
                    "<indoor-plan-id>13787134433</indoor-plan-id>",
                    "</company>",
                    "</companies>"});
}

BOOST_AUTO_TEST_CASE(entrances_yson_formatter_test)
{
    std::ostringstream out;
    PoiDef def1;
    {
        def1.id = 100500;
        def1.rubricIds.insert(42);
        def1.actualizationDate = maps::chrono::TimePoint::clock::now();
        def1.lon = 72.2;
        def1.lat = 74.8;
        def1.dispClass = 0;
        def1.names = {{"en", "english name"}, {"ru", "russian name"}};
        def1.shortNames
            = {{"en", "english shortname"}, {"ru", "russian shortname"}};
        def1.phones = {"(555) 555-55-55", "(666) 666-66-66"};
        def1.permalink = "123";
        def1.entrances = {{100501, 72.1, 74.7}, {100502, 72.3, 74.9}};
        def1.indoorPlanId = 13787134433;
    }
    PoiDef def2;
    {
        def2.id = 100501;
        def2.permalink = "456";
        def2.entrances = std::vector<EntranceDef>{};
    }
    PoiDef def3;
    {
        def3.id = 100502;
        def3.permalink = "789";
    }
    {
        EntrancesYsonFormatter formatter(out);
        formatter.append(def1);
        formatter.append(def2);
        formatter.append(def3);
    }
    const auto result = out.str();
    BOOST_CHECK_EQUAL(
        result,
        "[{"
            "\"originalId\":100500,"
            "\"permalink\":\"123\","
            "\"entrances\":[{"
                "\"lon\":72.1,"
                "\"lat\":74.7,"
                "\"type\":\"other\""
            "},{"
                "\"lon\":72.3,"
                "\"lat\":74.9,"
                "\"type\":\"other\""
            "}]"
        "},{"
            "\"originalId\":100501,"
            "\"permalink\":\"456\","
            "\"entrances\":[]"
        "}]");
}
