#include "config.h"
#include "export_poi.h"
#include "publish.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/tasks/export.h>
#include <yandex/maps/wiki/tasks/status_writer.h>
#include <maps/wikimap/mapspro/libs/rubrics/include/mapping_utils.h>
#include <library/cpp/resource/resource.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <filesystem>
#include <memory>
#include <sstream>

using maps::wiki::common::ExtendedXmlDoc;
using maps::wiki::common::RetryDurationPolicy;

namespace fs = std::filesystem;

namespace {

void runExportPoi(const ExtendedXmlDoc& configXml, const std::optional<std::string>& statusFilename)
{
    using namespace maps::wiki;

    maps::wiki::tasks::StatusWriter statusWriter(statusFilename);
    try {
        auto mapping = NResource::Find(rubrics::RUBRICS_MAPPING_CONFIG_RESOURCE_ID);
        poi::Config cfg{configXml, mapping};

        // Exclusive execution beetween hosts
        //
        maps::pgp3utils::PgAdvisoryXactMutex dbLocker(
            cfg.socialPool(),
            static_cast<int64_t>(maps::wiki::common::AdvisoryLockIds::EXPORT_POI));
        if (!dbLocker.try_lock()) {
            INFO() << "Database is already locked. Export POI task interrupted.";
            return;
        }

        cfg.setYMapsDfConnStr(tasks::detectNearestExportDatabase(configXml));

        INFO() << "Export POI task started";
        poi::exportPoi(cfg, [](const poi::Config& cfg) {
            publish(cfg, mds_dataset::DatasetStatus::Available,
                    {
                        cfg.resultFilePathCalculatedRubricId(),
                        cfg.md5FilePathCalculatedRubricId(),
                        cfg.resultFilePathAccurateRubricId(),
                        cfg.md5FilePathAccurateRubricId(),
                        cfg.resultFilePathVerifiedCoordsFeed(),
                        cfg.md5FilePathVerifiedCoordsFeed(),
                        cfg.resultFilePathNamesFeed(),
                        cfg.md5FilePathNamesFeed(),
                        cfg.resultFilePathCoordsFeed(),
                        cfg.md5FilePathCoordsFeed(),
                        cfg.resultFilePathIndoorFeed(),
                        cfg.md5FilePathIndoorFeed(),
                        cfg.resultFilePathIndoorNamesFeed(),
                        cfg.md5FilePathIndoorNamesFeed(),
                        cfg.resultFilePathEntrancesFeed(),
                        cfg.md5FilePathEntrancesFeed(),
                    });
        });
        INFO() << "Export POI task finished";
        statusWriter.flush();
    } catch (const std::exception& e) {
        ERROR() << "Export POI task failed: " << e.what();
        statusWriter.err(e.what());
        statusWriter.flush();
        throw;
    }
}

} // anonymous namespace

int main(int argc, char* argv[])
{
    maps::cmdline::Parser parser;
    auto workerConfig
        = parser.string("config").help("path to the worker configuration");
    auto syslogTag
        = parser.string("syslog-tag")
              .help("redirect log output to syslog with given tag");
    auto statusDir = parser.option<fs::path>("status-dir")
                             .help("path to status dir");

    parser.parse(argc, argv);

    RetryDurationPolicy::setDefaultDuration(std::chrono::minutes(15));

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    std::optional<std::string> statusFilename;
    if (statusDir.defined()) {
        fs::path filepath(statusDir);
        filepath /= "wiki-export-poi-worker.status";
        statusFilename = filepath;
    }

    try {
        std::unique_ptr<ExtendedXmlDoc> configDocPtr;
        if (workerConfig.defined()) {
            configDocPtr.reset(new ExtendedXmlDoc(workerConfig));
        }
        else {
            configDocPtr = maps::wiki::common::loadDefaultConfig();
        }

        runExportPoi(*configDocPtr, statusFilename);
        return EXIT_SUCCESS;
    }
    catch (const maps::Exception& e) {
        WARN() << e;

    }
    catch (const std::exception& e) {
        WARN() << e.what();
    }
    return EXIT_FAILURE;
}
