#include "feedback.h"

#include "message.h"

#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/log8/include/log8.h>

namespace fb = maps::wiki::social::feedback;

namespace maps::wiki::merge_poi {
namespace {
const auto FEEDBACK_SOURCE = "merge-poi-worker"s;
const auto MISSING_INDOOR_POI_SOURCE = "sprav-indoor"s;
const auto POI_VERIFICATION_SOURCE = "poi-verification"s;
const auto FEEDBACK_TASKS = "/feedback/tasks/"s;

namespace task_json {
const auto AFTER = "after"s;
const auto BEFORE = "before"s;
const auto BOUNDS = "bounds"s;
const auto CONTENT = "content"s;
const auto HIDDEN = "hidden"s;
const auto ERROR = "error"s;
const auto GEOMETRY = "geometry"s;
const auto MESSAGE = "message"s;
const auto MODIFIED = "modified"s;
const auto OBJECT_DIFF = "objectDiff"s;
const auto OBJECT_ID = "objectId"s;
const auto PERMALINK = "permalink"s;
const auto POSITION = "position"s;
const auto SOURCE = "source"s;
const auto SOURCE_CONTEXT = "sourceContext"s;
const auto SPRAV_FEED = "sprav-feed"s;
const auto STATUS = "status"s;
const auto TYPE = "type"s;
const auto WORKFLOW = "workflow"s;
} // task_json

geolib3::Point2 center(const geolib3::SimpleGeometryVariant& variant)
{
    return variant.visit(
        [](const auto& geometry) {
            return geometry.boundingBox().center();
        });
}

geolib3::BoundingBox bbox(const geolib3::SimpleGeometryVariant& variant)
{
    return variant.visit(
        [](const auto& geometry) {
            return geometry.boundingBox();
        });
}

geolib3::BoundingBox bbox(
    const poi_feed::FeedObjectData::Position& pos,
    const geolib3::SimpleGeometryVariant& variant)
{
    auto bb = bbox(variant);
    return geolib3::expand(bb, geolib3::Point2(pos.lon, pos.lat));
}

void fillCommonTaskJson(
    json::ObjectBuilder& taskBuilder,
    poi_feed::ObjectId oid,
    const geolib3::SimpleGeometryVariant& position,
    poi_feed::PermalinkId permalink,
    const std::string& source
    )
{
    taskBuilder[task_json::SOURCE] = source;
    taskBuilder[task_json::WORKFLOW] = toString(fb::Workflow::Task);
    if (oid) {
        taskBuilder[task_json::OBJECT_ID] = std::to_string(oid);
    }
    taskBuilder[task_json::HIDDEN] = true;
    taskBuilder[task_json::POSITION] = geolib3::geojson(center(position));
    taskBuilder[task_json::SOURCE_CONTEXT] << [&](json::ObjectBuilder context) {
        context[task_json::TYPE] = task_json::SPRAV_FEED;
        context[task_json::CONTENT] << [&](json::ObjectBuilder content) {
            content[task_json::PERMALINK] =  std::to_string(permalink);
        };
    };
}

void fillBoundsJson(
    json::ObjectBuilder& diffBuilder,
    const geolib3::BoundingBox& bounds)
{
    diffBuilder[task_json::BOUNDS] << [&](json::ArrayBuilder arrayBuilder) {
        arrayBuilder << bounds.minX();
        arrayBuilder << bounds.minY();
        arrayBuilder << bounds.maxX();
        arrayBuilder << bounds.maxY();
    };
}
} // namespace

std::string
prepareMovePoiTaskJsonBody(
    const poi_feed::FeedObjectData& patchData,
    const editor_client::BasicEditorObject& editorData)
{
    json::Builder jsonBuilder;
    const auto position = *patchData.position();
    jsonBuilder << [&](json::ObjectBuilder taskBuilder) {
        fillCommonTaskJson(
            taskBuilder,
            editorData.id,
            editorData.getGeometryInGeodetic().value(),
            patchData.permalink(),
            FEEDBACK_SOURCE);
        taskBuilder[fb::attrs::SUGGESTED_ACTION] = toString(fb::SuggestedAction::Modify);

        taskBuilder[task_json::OBJECT_DIFF] << [&](json::ObjectBuilder diffBuilder) {
            const auto bounds = bbox(position, editorData.getGeometryInGeodetic().value());
            fillBoundsJson(diffBuilder, bounds);
            diffBuilder[task_json::MODIFIED] << [&](json::ObjectBuilder modifiedBuilder) {
                modifiedBuilder[task_json::GEOMETRY] << [&](json::ObjectBuilder geomDiffBuilder) {
                    geomDiffBuilder[task_json::BEFORE] << [&](json::ArrayBuilder beforeGeoms) {
                        beforeGeoms << geolib3::geojson(editorData.getGeometryInGeodetic().value());
                    };
                    geomDiffBuilder[task_json::AFTER] << [&](json::ArrayBuilder afterGeoms) {
                        afterGeoms << geolib3::geojson(geolib3::Point2(position.lon, position.lat));
                    };
                };
            };
        };
    };
    return jsonBuilder.str();
}

std::string
prepareDeletePoiTaskJsonBody(
    const poi_feed::FeedObjectData& patchData,
    const editor_client::BasicEditorObject& editorData)
{
    json::Builder jsonBuilder;
    jsonBuilder << [&](json::ObjectBuilder taskBuilder) {
        fillCommonTaskJson(
            taskBuilder,
            editorData.id,
            editorData.getGeometryInGeodetic().value(),
            patchData.permalink(),
            FEEDBACK_SOURCE);
        taskBuilder[fb::attrs::SUGGESTED_ACTION] = toString(fb::SuggestedAction::Delete);
        taskBuilder[task_json::OBJECT_DIFF] << [&](json::ObjectBuilder diffBuilder) {
            const auto bounds = bbox(editorData.getGeometryInGeodetic().value());
            fillBoundsJson(diffBuilder, bounds);
            diffBuilder[task_json::MODIFIED] << [&](json::ObjectBuilder modifiedBuilder) {
                modifiedBuilder[task_json::GEOMETRY] << [&](json::ObjectBuilder geomDiffBuilder) {
                    geomDiffBuilder[task_json::BEFORE] << [&](json::ArrayBuilder beforeGeoms) {
                        beforeGeoms << geolib3::geojson(editorData.getGeometryInGeodetic().value());
                    };
                };
            };
        };
    };
    return jsonBuilder.str();
}

std::string
prepareCreateIndoorPoiTaskJsonBody(const poi_feed::FeedObjectData& candidate)
{
    geolib3::Point2 pos(candidate.position()->lon, candidate.position()->lat);
    json::Builder jsonBuilder;
    jsonBuilder << [&](json::ObjectBuilder taskBuilder) {
        fillCommonTaskJson(
            taskBuilder,
            0,
            pos,
            candidate.permalink(),
            MISSING_INDOOR_POI_SOURCE);
        taskBuilder[fb::attrs::SUGGESTED_ACTION] = toString(fb::SuggestedAction::CreateIndoorPoi);
        taskBuilder[task_json::OBJECT_DIFF] << [&](json::ObjectBuilder diffBuilder) {
            const auto bounds = bbox(pos);
            fillBoundsJson(diffBuilder, bounds);
            diffBuilder[task_json::MODIFIED] << [&](json::ObjectBuilder modifiedBuilder) {
                modifiedBuilder[task_json::GEOMETRY] << [&](json::ObjectBuilder geomDiffBuilder) {
                    geomDiffBuilder[task_json::AFTER] << [&](json::ArrayBuilder afterGeoms) {
                        afterGeoms << geolib3::geojson(pos);
                    };
                };
            };
        };
    };
    return jsonBuilder.str();
}

std::string
prepareVerifyPositionTaskJsonBody(const poi_feed::FeedObjectData& candidate)
{
    geolib3::Point2 pos(candidate.position()->lon, candidate.position()->lat);
    json::Builder jsonBuilder;
    jsonBuilder << [&](json::ObjectBuilder taskBuilder) {
        fillCommonTaskJson(
            taskBuilder,
            candidate.nmapsId(),
            pos,
            candidate.permalink(),
            POI_VERIFICATION_SOURCE);
        taskBuilder[fb::attrs::SUGGESTED_ACTION] = toString(fb::SuggestedAction::VerifyPosition);
        taskBuilder[task_json::OBJECT_DIFF] << [&](json::ObjectBuilder diffBuilder) {
            const auto bounds = bbox(pos);
            fillBoundsJson(diffBuilder, bounds);
            diffBuilder[task_json::MODIFIED] << [&](json::ObjectBuilder modifiedBuilder) {
                modifiedBuilder[task_json::GEOMETRY] << [&](json::ObjectBuilder geomDiffBuilder) {
                    geomDiffBuilder[task_json::AFTER] << [&](json::ArrayBuilder afterGeoms) {
                        afterGeoms << geolib3::geojson(pos);
                    };
                };
            };
        };
    };
    return jsonBuilder.str();
}

FeedbackClient::FeedbackClient(std::string feedbackApiUrl)
    : apiUrl_(std::move(feedbackApiUrl))
{
}

bool isFeedbackTaskCreated(const std::string& jsonString)
{
    const auto json = json::Value::fromString(jsonString);
    if (json.hasField(task_json::ERROR)) {
        const auto& error = json[task_json::ERROR];
        WARN() << "Failed to submit task " << error[task_json::STATUS].toString()
                  << ", message: " << error[task_json::MESSAGE];
        return false;
    }
    return true;
}

PatchResult
FeedbackClient::submitTask(std::string body)
{
    try {
        http::Request request(httpClient_, http::POST, apiUrl_ +
                FEEDBACK_TASKS + std::string{toString(fb::Type::Poi)});
        if (tvmTicketProvider_) {
            request.addHeader(auth::SERVICE_TICKET_HEADER, tvmTicketProvider_());
        }
        request.setContent(body);
        auto response = request.perform();
        const auto responseBody = response.readBody();
        if (response.status() == HTTP_OK && isFeedbackTaskCreated(responseBody)) {
            return { Resolution::CreatedFeedback, {} };
        }
        ERROR() << "Failed to create feedback task:" << body
            << "\n response: " << responseBody;
    } catch (const std::exception& ex) {
        ERROR() << "Failed to create feedback task:" << ex.what() << "\n" << body;
    }
    return { Resolution::Failed, {} };
}


PatchResult
FeedbackClient::createMoveTask(
    const poi_feed::FeedObjectData& patchData,
    const editor_client::BasicEditorObject& editorData)
{
    return submitTask(prepareMovePoiTaskJsonBody(patchData, editorData));
}

PatchResult
FeedbackClient::createDeleteTask(
    const poi_feed::FeedObjectData& patchData,
    const editor_client::BasicEditorObject& editorData)
{
    return submitTask(prepareDeletePoiTaskJsonBody(patchData, editorData));
}

PatchResult
FeedbackClient::createMissingIndoorPoiTask(const poi_feed::FeedObjectData& candidate)
{
    return submitTask(prepareCreateIndoorPoiTaskJsonBody(candidate));
}

PatchResult
FeedbackClient::createVerifyPositionTask(const poi_feed::FeedObjectData& candidate)
{
    return submitTask(prepareVerifyPositionTaskJsonBody(candidate));
}
} // namespace maps::wiki::merge_poi
