#include "object_helpers.h"
#include "message.h"

#include <maps/wikimap/mapspro/libs/editor_client/include/exception.h>
#include <maps/wikimap/mapspro/libs/poi_feed/include/helpers.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/robot.h>

namespace maps::wiki::merge_poi {

namespace {
const std::string DISP_CLASS_SUFFIX = ":disp_class";
const std::string BUSINESS_ID_SUFFIX = ":business_id";
const std::string POI_IS_GEOPRODUCT = "poi:is_geoproduct";
const std::string POI_POSITION_QUALITY = "poi:position_quality";
const std::string BUSINESS_RUBRIC_ID_SUFFIX = ":business_rubric_id";
const std::string PROTECTED_SUFFIX = ":protected";
const std::string STR_TRUE = "1";
const std::string STR_FALSE = "";

namespace category {
const std::string ADDR = "addr";
const std::string BLD = "bld";
const std::string POI_ENTRANCE = "poi_entrance";
const std::string POI_PREFIX = "poi";
const std::string INDOOR_POI_PREFIX = "indoor_poi";
} // category

const double CHECK_DISTANCE_METERS = 2.0;
const double CHECK_BLD_HIT_BOX_METERS = 0.0005;
const double CHECK_OBJECTS_LIMIT = 100;

std::vector<editor_client::ObjectIdentity>
getBlds(const editor_client::Instance& editorInstance, const geolib3::SimpleGeometryVariant& geometry)
{
    return editorInstance.getObjectsByLasso(
            {category::BLD},
            geolib3::convertGeodeticToMercator(geometry),
            CHECK_BLD_HIT_BOX_METERS,
            CHECK_OBJECTS_LIMIT,
            editor_client::GeomPredicate::Intersects);
}
} // namespace

std::optional<DispClass>
dispClass(const editor_client::BasicEditorObject& object)
{
    for (const auto& [name, value] : object.plainAttributes) {
        if (!name.ends_with(DISP_CLASS_SUFFIX)) {
            continue;
        }
        return value.empty()
            ? std::nullopt
            : std::optional<DispClass>(boost::lexical_cast<DispClass>(value));
    }
    return std::nullopt;
}

std::optional<poi_feed::PermalinkId>
permalink(const editor_client::BasicEditorObject& object)
{
    for (const auto& [name, value] : object.plainAttributes) {
        if (!name.ends_with(BUSINESS_ID_SUFFIX)) {
            continue;
        }
        return value.empty()
            ? std::nullopt
            : std::optional<poi_feed::PermalinkId>(
                boost::lexical_cast<poi_feed::PermalinkId>(value));
    }
    return std::nullopt;
}

std::optional<poi_feed::RubricId>
rubricId(const editor_client::BasicEditorObject& object)
{
    for (const auto& [name, value] : object.plainAttributes) {
        if (!name.ends_with(BUSINESS_RUBRIC_ID_SUFFIX)) {
            continue;
        }
        return value.empty()
            ? std::nullopt
            : std::optional<poi_feed::RubricId>(
                boost::lexical_cast<poi_feed::RubricId>(value));
    }
    return std::nullopt;
}

void setPermalink(editor_client::BasicEditorObject& object, poi_feed::PermalinkId permalink)
{
    std::string attrPrefix = category::POI_PREFIX;
    if (!object.categoryId.starts_with(category::POI_PREFIX) &&
        !object.categoryId.starts_with(category::INDOOR_POI_PREFIX)) {
        attrPrefix = object.categoryId;
    }
    object.plainAttributes[attrPrefix + BUSINESS_ID_SUFFIX] = std::to_string(permalink);
}

bool hasVerifiedCoordinates(const editor_client::BasicEditorObject& object)
{
    return object.plainAttributes.contains(POI_POSITION_QUALITY) &&
        poi_feed::isVerifiedPositionQuality(
            object.plainAttributes.at(POI_POSITION_QUALITY));
}

bool isProtected(const editor_client::BasicEditorObject& object)
{
    for (const auto& [name, value] : object.plainAttributes) {
        if (name.ends_with(PROTECTED_SUFFIX)) {
            return value == STR_TRUE;
        }
    }
    return false;
}

bool
isRecentCommitByRobot(const editor_client::BasicEditorObject& object)
{
    return
        object.recentCommit &&
        object.recentCommit->author == maps::wiki::common::WIKIMAPS_SPRAV_UID;
}

std::optional<size_t>
recentCommitAgeDays(const editor_client::BasicEditorObject& object)
{
    if (!object.recentCommit) {
        return std::nullopt;
    }
    auto now = maps::chrono::TimePoint::clock::now();
    return std::chrono::duration_cast<std::chrono::hours>(
        now - object.recentCommit->date).count() / 24;
}

bool
isSameBuildingPatch(
    const editor_client::Instance& editorInstance,
    const poi_feed::FeedObjectData& patch)
{
    try {
        const auto editorObject = editorInstance.getObjectById(patch.nmapsId());
        if (editorObject.deleted) {
            return false;
        }
        const auto bldsAtSource = getBlds(editorInstance, editorObject.getGeometryInGeodetic().value());
        if (bldsAtSource.empty()) {
            return false;
        }
        const auto bldsAtDst = getBlds(editorInstance,
            geolib3::Point2(patch.position()->lon, patch.position()->lat));
        if (bldsAtDst.empty()) {
            return false;
        }
        std::set<poi_feed::ObjectId> bldsAtDestination;
        for (const auto& bldAtDst : bldsAtDst) {
            bldsAtDestination.insert(bldAtDst.id);
        }
        for (const auto& bldAtSource : bldsAtSource) {
            if (!bldsAtDestination.count(bldAtSource.id)) {
                return false;
            }
        }
        return true;
    } catch (const editor_client::ServerException& ex) {
        WARN() << "isSameBuildingPatch error: "
            << patch.nmapsId() << " " << ex.status() << " " << ex.serverResponse();
    } catch (const maps::Exception& ex) {
        WARN() << "isSameBuildingPatch error: "
            << patch.nmapsId() << " " << ex;
    } catch (const std::exception& ex) {
        WARN() << "isSameBuildingPatch error: "
            << patch.nmapsId() << " " << ex.what();
    } catch (...) {
        WARN() << "isSameBuildingPatch unknown error: "
            << patch.nmapsId();
    }
    return false;
}

RelocationResult checkRelocationValidity(
    const editor_client::Instance& editorInstance,
    const editor_client::BasicEditorObject& object,
    const poi_feed::FeedObjectData& patch)
{
    RelocationResult result;
    const auto newPositionMercator =
        geolib3::convertGeodeticToMercator(geolib3::Point2(
            patch.position()->lon,
            patch.position()->lat));
    result.poiConflict = poiConflictsPriority(
        editorInstance,
        newPositionMercator,
        object.id,
        isGeoproduct(object)
            ? poi_feed::FeedObjectData::IsGeoproduct::Yes
            : poi_feed::FeedObjectData::IsGeoproduct::No);
    const auto objectsAtDestination = editorInstance.getObjectsByLasso(
        {
            category::ADDR,
            category::POI_ENTRANCE
        },
        newPositionMercator,
        CHECK_DISTANCE_METERS,
        CHECK_OBJECTS_LIMIT,
        editor_client::GeomPredicate::Intersects);

    result.addrCollision = AddrCollision::False;
    result.entranceCollision = EntranceCollision::False;
    for (const auto& objectAtDestination : objectsAtDestination) {
        const auto& dstCategoryId = objectAtDestination.categoryId;
        if (dstCategoryId == category::ADDR) {
            result.addrCollision = AddrCollision::True;
        } else if (dstCategoryId == category::POI_ENTRANCE) {
            result.entranceCollision = EntranceCollision::True;
        }
        if (result.addrCollision == AddrCollision::True &&
            result.entranceCollision == EntranceCollision::True) {
            break;
        }
    }


    const auto bldsAtDst = getBlds(
        editorInstance,
        geolib3::Point2(patch.position()->lon, patch.position()->lat));
    std::set<poi_feed::ObjectId> bldsAtDestination;
    for (const auto& bldAtDst : bldsAtDst) {
        bldsAtDestination.insert(bldAtDst.id);
    }
    const auto bldsAtSource = getBlds(
        editorInstance,
        object.getGeometryInGeodetic().value());
    result.bldChange = BldChange::False;
    if (!bldsAtSource.empty() && bldsAtSource.size() != bldsAtDestination.size()) {
        result.bldChange = BldChange::True;
        result.sameBuilding = SameBuilding::False;
    } else {
        for (const auto& bldAtSource : bldsAtSource) {
            if (!bldsAtDestination.count(bldAtSource.id)) {
                result.bldChange = BldChange::True;
                result.sameBuilding = SameBuilding::False;
                break;
            }
        }
    }
    if (result.bldChange == BldChange::False) {
        result.sameBuilding =
            bldsAtSource.empty()
                ? SameBuilding::False
                : SameBuilding::True;
    }
    return result;
}

bool
valid(const RelocationResult& relocationResult)
{
    return
        relocationResult.bldChange != BldChange::True &&
        relocationResult.addrCollision != AddrCollision::True;
}

std::set<std::string>
createMessages(const RelocationResult& relocationResult)
{
    std::set<std::string> messages;
    if (relocationResult.addrCollision == AddrCollision::True) {
        messages.insert(message::MOVE_ADDR_COLLISION);
    }
    if (relocationResult.bldChange == BldChange::True) {
        messages.insert(message::MOVE_BLD_CHANGE);
    }
    if (relocationResult.sameBuilding == SameBuilding::True) {
        messages.insert(message::MOVE_SAME_BUILDING);
    }
    if (relocationResult.entranceCollision == EntranceCollision::True) {
        messages.insert(message::MOVE_ENTRANCE_COLLISION);
    }
    if (relocationResult.poiConflict == PoiConflict::High) {
        messages.insert(message::SEVERE_POI_CONFLICT);
    }
    if (relocationResult.poiConflict == PoiConflict::Low) {
        messages.insert(message::POI_CONFLICT);
    }
    return messages;
}

std::string toJson(const std::set<std::string>& messages)
{
    json::Builder builder;
    builder << [&](json::ArrayBuilder array) {
        for (const auto& message : messages) {
            array << message;
        }
    };
    return builder.str();
}

bool hasIsGeoproductAttr(const editor_client::BasicEditorObject& object)
{
    return
        object.plainAttributes.count(POI_IS_GEOPRODUCT);
}

bool isGeoproduct(const editor_client::BasicEditorObject& object)
{
    return
        object.plainAttributes.count(POI_IS_GEOPRODUCT) &&
        object.plainAttributes.at(POI_IS_GEOPRODUCT) == STR_TRUE;
}

void setIsGeoproduct(
    editor_client::BasicEditorObject& object,
    const poi_feed::FeedObjectData::IsGeoproduct newGeoproductValue)
{
    if (newGeoproductValue ==
        poi_feed::FeedObjectData::IsGeoproduct::No) {
        object.plainAttributes[POI_IS_GEOPRODUCT] = STR_FALSE;
    } else {
        object.plainAttributes[POI_IS_GEOPRODUCT] = STR_TRUE;
    }
}

std::optional<editor_client::BasicEditorObject>
getEditorObject(const editor_client::Instance& editor, poi_feed::ObjectId oid)
try {
    return  editor.getObjectById(oid);
} catch (editor_client::ServerException& ex) {
    WARN() << "Editor backend reported error:" << ex;
    return std::nullopt;
}

std::optional<chrono::TimePoint>
getEditorObjectModificationDate(const editor_client::Instance& editor, poi_feed::ObjectId oid)
{
    try {
        const auto history = editor.getHistory(oid, 1, 1);
        if (!history.empty()) {
            return history[0].date;
        }
    } catch (editor_client::ServerException& ex) {
        WARN() << "Editor backend reported error:" << ex;
    }
    return std::nullopt;
}

PoiConflict
poiConflictsPriority(
    const editor_client::Instance& editor,
    const geolib3::Point2& newPositionMercator,
    poi_feed::ObjectId selfId,
    poi_feed::FeedObjectData::IsGeoproduct selfIsGeoproduct)
{
    auto conflicts = editor.getPoiConflicts(
        selfId,
        {},
        newPositionMercator,
        selfIsGeoproduct == poi_feed::FeedObjectData::IsGeoproduct::Yes
            ? editor_client::IsGeoproduct::True
            : editor_client::IsGeoproduct::False);
    if (conflicts.zoomToConflictingObjects.empty()) {
        return PoiConflict::NoData;
    }
    for (const auto& [_, severityToObjects] : conflicts.zoomToConflictingObjects) {
        if (severityToObjects.count(poi_conflicts::ConflictSeverity::Critical)) {
            return PoiConflict::High;
        }
    }
    return PoiConflict::Low;
}

} // namespace maps::wiki::merge_poi
