#include "sync_geoproduct_flag.h"

#include "altay_reader.h"
#include "config.h"
#include "nmaps_export_result.h"
#include "object_helpers.h"

#include <yandex/maps/wiki/common/robot.h>
#include <maps/wikimap/mapspro/libs/editor_client/include/exception.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <yandex/maps/wiki/threadutils/executor.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/make_batches.h>

namespace maps::wiki::merge_poi {

namespace {
const size_t THREADS_COUNT = 6;
const size_t BATCH_SIZE = 1000;
const uint32_t HIGH_CONFLICTS_MESSAGE_PRIORITY = 0;
const uint32_t LOW_CONFLICTS_MESSAGE_PRIORITY = 2;

struct EditorInstances
{
    EditorInstances(const Config& cfg)
    : editorInstance(cfg.editorWriterUrl(), common::WIKIMAPS_SPRAV_UID)
    , editorModeratedInstance(cfg.editorWriterUrl(), common::WIKIMAPS_SPRAV_COMMON_UID)
    {}

    editor_client::Instance editorInstance;
    editor_client::Instance editorModeratedInstance;

};

void updateGeoproductFlag(
    poi_feed::ObjectId nmapsId,
    EditorInstances& editorInstances,
    const poi_feed::FeedObjectData::IsGeoproduct newGeoproductValue)
{
    auto& editorInstance = editorInstances.editorInstance;
    try {
        auto editorObject = getEditorObject(editorInstance, nmapsId);
        if (!editorObject) {
            return;
        }
        if (!hasIsGeoproductAttr(*editorObject)) {
            WARN() << "Object doesn't provide is_geoproduct attribute:" << nmapsId;
            return;
        }
        if (isGeoproduct(*editorObject) ==
            (poi_feed::FeedObjectData::IsGeoproduct::Yes == newGeoproductValue))
        {
            return;
        }
        setIsGeoproduct(*editorObject, newGeoproductValue);
        const auto position = editorObject->getGeometryInMercator()->get<geolib3::Point2>();
        PoiConflict poiConflictLevel = PoiConflict::NoData;
        if (false && // Skip moderation on conflicts with newly set flag. NMAPS-13205
            poi_feed::FeedObjectData::IsGeoproduct::Yes == newGeoproductValue &&
            PoiConflict::NoData !=
                (poiConflictLevel = poiConflictsPriority(
                    editorInstance,
                    position,
                    editorObject->id,
                    poi_feed::FeedObjectData::IsGeoproduct::Yes)))
        {
            editorInstances.editorModeratedInstance.saveObject(
                *editorObject,
                {
                    {
                        poiConflictLevel == PoiConflict::High
                            ? HIGH_CONFLICTS_MESSAGE_PRIORITY
                            : LOW_CONFLICTS_MESSAGE_PRIORITY,
                        POI_CONFLICTS_DIFFALERT_MESSAGE
                    }
                });
        } else {
            editorInstance.saveObject(*editorObject);
        }
        INFO() << "Set oid: "
                << nmapsId << " is_geoproduct: "
                << (poi_feed::FeedObjectData::IsGeoproduct::Yes == newGeoproductValue);
    } catch (const editor_client::ServerException& ex) {
        WARN() << "Editor backend reported error while updating is_geoproduct for object:"
            << nmapsId << " " << ex.status() << "\n" << ex;
    } catch (const maps::Exception& ex) {
        ERROR() << "Exception while updating is_geoproduct for object: "
            << nmapsId << "\n" << ex;
    } catch (const std::exception& ex) {
        ERROR() << "Exception while updating is_geoproduct for object: "
            << nmapsId << "\n" << ex.what();
    }
}

void processIdsBatch(
    const maps::common::Batch<std::set<poi_feed::ObjectId>>& ids,
    const poi_feed::FeedObjectData::IsGeoproduct newGeoproductValue,
    const Config& cfg)
{
    EditorInstances editorInstances(cfg);
    for (const auto& nmapsId : ids) {
        updateGeoproductFlag(nmapsId, editorInstances, newGeoproductValue);
    }
}

} // namespace

void syncGeoproductFlag(
    const NmapsExportResult& recentExportData,
    const Config& cfg)
{
    const auto currentGeoproductPermalinks = readGeoproductPermalinks();
    //create list to set flag
    std::set<poi_feed::ObjectId> toSetFlag;
    for (const auto& permalink : currentGeoproductPermalinks) {
        const auto objectDatas = recentExportData.findByPermalink(permalink);
        for (const auto objectData : objectDatas) {
            if (!objectData->isGeoproduct()) {
                toSetFlag.insert(objectData->nmapsId());
            }
        }
    }
    ThreadPool threadPool(THREADS_COUNT);
    Executor threadedExecutor;
    auto batches = maps::common::makeBatches(toSetFlag, BATCH_SIZE);
    for (const auto& batch : batches) {
        threadedExecutor.addTask([&, batch] {
            processIdsBatch(
                batch,
                poi_feed::FeedObjectData::IsGeoproduct::Yes,
                cfg);
        });
    };
    threadedExecutor.executeAllInThreads(threadPool);

    //create list to remove flag
    std::set<poi_feed::ObjectId> toRemoveFlag;
    for (const auto& objectData : recentExportData) {
        if (objectData.isGeoproduct() &&
            !currentGeoproductPermalinks.count(objectData.permalink())) {
            toRemoveFlag.insert(objectData.nmapsId());
        }
    }
    batches = maps::common::makeBatches(toRemoveFlag, BATCH_SIZE);
    for (const auto& batch : batches) {
        threadedExecutor.addTask([&, batch] {
            processIdsBatch(
                batch,
                poi_feed::FeedObjectData::IsGeoproduct::No,
                cfg);
        });
    };
    threadedExecutor.executeAllInThreads(threadPool);

}
} // maps::wiki::merge_poi
