#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <maps/wikimap/mapspro/services/tasks_sprav/src/merge_poi_worker/config.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/merge_poi_worker/names.h>
#include <maps/wikimap/mapspro/libs/editor_client/impl/parser.h>

#include <maps/libs/common/include/file_utils.h>

using namespace maps::wiki::editor_client;
using namespace maps::wiki::poi_feed;
using namespace std::string_literals;

namespace maps::wiki::merge_poi {
namespace {
const std::string patchDataJson =
    R"(
        {
            "id": "0",
            "lon": 33.083493527,
            "lat": 68.903090685,
            "names": [
                {"lang":"en", "name": "wood factory"},
                {"lang":"en", "name": "Driving school"},
                {"lang":"en", "name": "factory1"}
                ],
            "revision": "1520450407083",
            "toDelete": false,
            "permalink": "0",
            "shortNames": [
                {"lang":"en", "name": "factory"},
                {"lang":"en", "name": "factory1"}
                ],
            "ftTypeId": 691
        }
    )";

const std::string poiDataJson =
    R"(
        {
            "id": "1541673983",
            "revisionId": "1541673983:28236989",
            "categoryId": "poi_finance",
            "title": "Банк Открытие, банкомат",
            "geometry": {
                "type": "Point",
                "coordinates": [
                    37.588356297,
                    55.734719967
                ]
            },
            "state": "draft",
            "attrs": {
                "poi_nm": [
                    {
                        "poi_nm:is_local": true,
                        "poi_nm:lang": "ru",
                        "poi_nm:name": "Банк Открытие, банкомат",
                        "poi_nm:name_type": "official"
                    },
                    {
                        "poi_nm:is_local": true,
                        "poi_nm:lang": "ru",
                        "poi_nm:name": "Открытие",
                        "poi_nm:name_type": "render_label"
                    },
                    {
                        "poi_nm:is_local": false,
                        "poi_nm:lang": "en",
                        "poi_nm:name": "moneyhole",
                        "poi_nm:name_type": "synonym"
                    }
                ],
                "poi_finance:ft_type_id": "1503",
                "poi_finance:addr_info": "",
                "poi_finance:open_hours": "пн-вс круглосуточно",
                "poi_finance:open_type": "1",
                "poi_finance:url": "openbank.ru, bank-open.ru",
                "poi_finance:email": "",
                "poi_finance:phone": "+7 (800) 700-78-77, +7 (495) 787-78-77",
                "poi_finance:disp_class": "7",
                "sys:import_source": "nmaps",
                "sys:import_source_id": "55379840",
                "poi:business_id": "1250333852",
                "poi:business_rubric_id": "",
                "sys:protected": false,
                "sys:blocked": false
            }
        }
    )";
const auto EDITOR_XML_PATH =
    ArcadiaSourceRoot() +
    "/maps/wikimap/mapspro/cfg/editor/editor.xml";

std::string readTestData(const std::string& fileName)
{
    return maps::common::readFileToString(SRC_("data") + "/" + fileName);
}
} // namespace

Y_UNIT_TEST_SUITE(Names)
{

Y_UNIT_TEST(IsRubric)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    UNIT_ASSERT(isRubricName({"en", "Driving school"}, editorCfg));
    UNIT_ASSERT(!isRubricName({"en", "Driving schol"}, editorCfg));
    UNIT_ASSERT(!isRubricName({"ru", "Driving school"}, editorCfg));
}

Y_UNIT_TEST(ReadPatchNames)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    FeedObjectData patch(patchDataJson);
    Names patchNames(patch);
    UNIT_ASSERT(patchNames.localLangs().empty());
    UNIT_ASSERT_EQUAL(patchNames.size(), 5);
}

Y_UNIT_TEST(ReadNmapsNames)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    auto poi = parseJsonResponse(poiDataJson);
    UNIT_ASSERT_EQUAL(guessNameAttr(poi, editorCfg), "poi_nm");
    Names poiNames(poi, editorCfg);
    UNIT_ASSERT_EQUAL(poiNames.localLangs(), std::unordered_set<Lang>{"ru"});
    UNIT_ASSERT_EQUAL(poiNames.size(), 3);
}

Y_UNIT_TEST(UpdateNamesNewLangName)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    FeedObjectData patch(readTestData("altay_1541971055.json"));
    auto poiFood = parseJsonResponse(readTestData("editor_1541971055.json"));
    Names nmapsNames(poiFood, editorCfg);
    auto updated = nmapsNames.update(Names(patch), editorCfg);
    UNIT_ASSERT_EQUAL(updated.size(), 1);
    UNIT_ASSERT(updated.count(Names::UpdateResult::Updated));
    nmapsNames.writeTo(poiFood, editorCfg);
    Names updatedNames(poiFood, editorCfg);
    UNIT_ASSERT(updatedNames.update(Names(patch), editorCfg).empty());
}

Y_UNIT_TEST(UpdateNamesNoChanges)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    FeedObjectData patch(readTestData("altay_1541991905.json"));
    auto poiFood = parseJsonResponse(readTestData("editor_1541991905.json"));
    Names nmapsNames(poiFood, editorCfg);
    UNIT_ASSERT(nmapsNames.update(Names(patch), editorCfg).empty());
}

Y_UNIT_TEST(UpdateNamesNeedModeration)
{
    const EditorCfg editorCfg(EDITOR_XML_PATH);
    FeedObjectData patch(readTestData("altay_2038590180.json"));
    auto poiFood = parseJsonResponse(readTestData("editor_2038590180.json"));
    Names nmapsNames(poiFood, editorCfg);
    UNIT_ASSERT(nmapsNames.update(Names(patch), editorCfg).size() > 1);
}

Y_UNIT_TEST(FixContradictingNamesLangs)
{
    Names patchHR(
        {
            Name {
                NameType::Official,
                IsLocal::No,
                "hr"s,
                "Приятно свэ найбольэ"s
            }
        });
    Names patchSR(
        {
            Name {
                NameType::Official,
                IsLocal::No,
                "sr"s,
                "Приятно свэ найбольэ"s
            }
        });
    Names objectSR(
        {
            Name {
                NameType::Official,
                IsLocal::No,
                "sr-latn"s,
                "Приятно свэ найбольэ"s
            }
        });

    const auto patchHrFixed = objectSR.fixContradictingNamesLangs(patchHR);
    UNIT_ASSERT_EQUAL(patchHrFixed.langs().size(), 1);
    UNIT_ASSERT_EQUAL(*patchHrFixed.langs().begin(), "sr-latn"s);

    const auto patchSRFixed = objectSR.fixContradictingNamesLangs(patchSR);
    UNIT_ASSERT_EQUAL(patchSRFixed.langs().size(), 1);
    UNIT_ASSERT_EQUAL(*patchSRFixed.langs().begin(), "sr"s);

}

} // Y_UNIT_TEST_SUITE(Names)
} // namespace maps::wiki::merge_poi
