#include <maps/wikimap/mapspro/services/tasks_sprav/src/walkers_export_downloader/lib/config.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/walkers_export_downloader/lib/process_queue.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/walkers_export_downloader/lib/sync_database.h>

#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/tasks/status_writer.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <filesystem>

using maps::wiki::common::ExtendedXmlDoc;

namespace fs = std::filesystem;
namespace wed = maps::wiki::walkers_export_downloader;

namespace {

void runDownloader(
    const ExtendedXmlDoc& configXml,
    const std::optional<std::string>& statusFilename)
{
    maps::wiki::tasks::StatusWriter statusWriter(statusFilename);
    try {
        wed::Config cfg {configXml};
        cfg.initTvm();

        maps::pgp3utils::PgAdvisoryXactMutex dbLocker(
            cfg.socialPool(),
            static_cast<int64_t>(maps::wiki::common::AdvisoryLockIds::WALKERS_EXPORT_DOWNLOADER));
        if (!dbLocker.try_lock()) {
            INFO() << "Database is already locked. Downloader task interrupted.";
            return;
        }

        INFO() << "Downloader task started";
        wed::syncDatabase(cfg);
        INFO() << "Downloader task finished";
        INFO() << "Processing photo bundles queue for feedback started.";
        wed::processQueue(cfg);
        INFO() << "Processing photo bundles queue for feedback finished.";

        statusWriter.flush();
    } catch (const std::exception& e) {
        ERROR() << "Downloader task failed: " << e.what();
        statusWriter.err(e.what());
        statusWriter.flush();
        throw;
    }
}

} // namespace

int main(int argc, char* argv[])
{
    maps::cmdline::Parser parser;
    auto workerConfig = parser.string("config")
        .help("path to the worker configuration");
    auto syslogTag = parser.string("syslog-tag")
        .help("redirect log output to syslog with given tag");
    auto statusDir = parser.option<fs::path>("status-dir")
        .help("path to status dir");

    parser.parse(argc, argv);
    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    std::optional<std::string> statusFilename;
    if (statusDir.defined()) {
        fs::path filepath(statusDir);
        filepath /= "wiki-walkers-export-downloader.status";
        statusFilename = filepath;
    }

    try {
        const auto configDocPtr = workerConfig.defined()
            ? std::make_unique<ExtendedXmlDoc>(workerConfig)
            : maps::wiki::common::loadDefaultConfig();
        runDownloader(*configDocPtr, statusFilename);
        return EXIT_SUCCESS;
    } catch (const maps::Exception& e) {
        ERROR() << e;
    } catch (const std::exception& e) {
        ERROR() << e.what();
    }

    return EXIT_FAILURE;
}
