#include "process_queue.h"
#include "process_queue_helpers.h"
#include "company_data_bundle_queue.h"

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/common.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature.h>
#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>

#include <maps/wikimap/mapspro/services/mrc/libs/common/include/exif.h>

#include <maps/wikimap/mapspro/libs/editor_client/include/basic_object.h>
#include <maps/wikimap/mapspro/libs/editor_client/include/instance.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <maps/wikimap/mapspro/libs/poi_feed/include/helpers.h>
#include <yandex/maps/wiki/common/robot.h>

#include <maps/libs/json/include/builder.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/log8/include/log8.h>

namespace editor = maps::wiki::editor_client;
namespace fb = maps::wiki::social::feedback;
namespace mrc_db = maps::mrc::db;
namespace mrc_common = maps::mrc::common;

namespace maps::wiki::walkers_export_downloader {

namespace {

const size_t MAX_BUNDLES = 10000;
const std::string ATTR_POI_POSITION_QUALITY = "poi:position_quality";
const std::string INDOOR_PREFIX = "indoor_";
const double POI_TO_SHOOTING_TARGET_THRESHOLD = 40.0; // meters
constexpr size_t THUMBNAIL_HEIGHT_PX = 160;
const std::string CREATE_POI_SOURCE = "experiment-walkers-poi-photo-create";
const std::string VERIFIED_POI_PHOTO_SOURCE = "walkers-poi-photo-verified-poi";
const std::string NOT_VERIFIED_POI_PHOTO_SOURCE = "walkers-poi-photo-not-verified-poi";
const std::string ALTAY_CARDS_URL = "https://altay.yandex-team.ru/cards/";
const std::string FEEDBACK_TASKS_API_BASE = "/feedback/tasks/";
const int HTTP_OK = 200;
const double PERMALINK_SEARCH_RADIUS = 300.0; // meters
} // namespace

namespace helpers {
double
mercatorCoordsDistaceMeters(const geolib3::Point2& p1, const geolib3::Point2& p2)
{
    return
        geolib3::geoDistance(
            geolib3::mercator2GeoPoint(p1),
            geolib3::mercator2GeoPoint(p2));
}

std::map<PhotoId, mrc_db::Feature>
loadBundleFeatures(const CompanyDataBundle& bundle, const Config& cfg)
{
    maps::mrc::db::TIdSet featureIds;
    for (const auto& photo : bundle.photos) {
        featureIds.insert(std::stoll(photo.photoId));
    }
    auto txn = cfg.mrcPool().masterReadOnlyTransaction();
    auto mrcFeatures = mrc_db::FeatureGateway(*txn).loadByIds(
            {featureIds.begin(), featureIds.end()});
    std::map<PhotoId, mrc_db::Feature> result;
    for (const auto& feature : mrcFeatures) {
        result.emplace(std::to_string(feature.id()), feature);
    }
    return result;
}

std::map<PhotoId, mrc_db::Feature>
removeNotPublished(std::map<PhotoId, mrc_db::Feature> mrcFeatures)
{
    for (auto it = mrcFeatures.begin(); it != mrcFeatures.end();) {
        if (!it->second.isPublished()) {
            it = mrcFeatures.erase(it);
        } else {
            ++it;
        }
    }
    return mrcFeatures;
}

bool
isVerifiedPoi(
    const editor::BasicEditorObject& nmapsObject)
{
    return nmapsObject.plainAttributes.contains(ATTR_POI_POSITION_QUALITY) &&
        poi_feed::isVerifiedPositionQuality(
            nmapsObject.plainAttributes.at(ATTR_POI_POSITION_QUALITY));
}

bool
isIndoorPoi(
    const editor::BasicEditorObject& nmapsObject)
{
    return nmapsObject.categoryId.starts_with(INDOOR_PREFIX);
}

double
maxDistancePoiToShootingTarget(
    const CompanyDataBundle& bundle,
    const editor::BasicEditorObject& nmapsObject)
{
    double maxDist = 0;
    const auto objGeom = nmapsObject.getGeometryInMercator()->get<geolib3::Point2>();
    for (const auto& photo : bundle.photos) {
        const auto distance = mercatorCoordsDistaceMeters(objGeom, photo.shootingTarget);
        if (distance > maxDist) {
            maxDist = distance;
        }
    }
    return maxDist;
}

bool
taskShouldBeCreated(
    const CompanyDataBundle& bundle,
    const editor::BasicEditorObject& nmapsObject)
{
    if (isIndoorPoi(nmapsObject)) {
        return false;
    }
    if (!isVerifiedPoi(nmapsObject)) {
        return true;
    }
    if (maxDistancePoiToShootingTarget(bundle, nmapsObject) > POI_TO_SHOOTING_TARGET_THRESHOLD) {
        return true;
    }
    // TODO if different blds return true
    return false;
}


mrc_db::FeaturePrivacy
evalMaxPrivacy(const std::map<PhotoId, mrc_db::Feature>& mrcFeatures)
{
    auto maxPrivacy = mrc_db::FeaturePrivacy::Public;
    for (const auto& [_, feature] : mrcFeatures) {
        maxPrivacy = std::max(maxPrivacy, feature.privacy());
        if (maxPrivacy == mrc_db::FeaturePrivacy::Max) {
            break;
        }
    }
    return maxPrivacy;
}

std::string
makeBaseImageUrl(const MrcUrls& urls, mrc_db::FeaturePrivacy privacy)
{
    if (privacy == mrc_db::FeaturePrivacy::Secret) {
        return urls.pro;
    }
    return urls.common;
}

std::string
makeFullImageUrl(const MrcUrls& urls, const mrc_db::Feature& feature)
{
    auto result = makeBaseImageUrl(urls, feature.privacy());
    return makeBaseImageUrl(urls, feature.privacy()) +
        "/feature/" + std::to_string(feature.id()) + "/image";
}

std::string
makeImagePreviewUrl(const MrcUrls& urls, const mrc_db::Feature& feature)
{
    return makeBaseImageUrl(urls, feature.privacy()) +
        "/feature/" + std::to_string(feature.id()) + "/thumbnail";
}

mrc_common::Size
thumbnailSize(const mrc_common::Size& size)
{
    return {size.width * THUMBNAIL_HEIGHT_PX / size.height,
            THUMBNAIL_HEIGHT_PX};
}

mrc_common::Size
normThumbnailSizeOf(const mrc_db::Feature& feature)
{
    const auto size
        = transformByImageOrientation(feature.size(), feature.orientation());
    return thumbnailSize(size);
}

void
formatBundleFeature(
    json::ObjectBuilder& imageFeature,
    const PhotoData& photo,
    const mrc_db::Feature& feature,
    const MrcUrls& mrcUrls)
{
    imageFeature["id"] = std::to_string(feature.id());
    imageFeature["heading"] = feature.heading().value();
    imageFeature["geometry"] = geolib3::geojson(geolib3::mercator2GeoPoint(photo.shootingPoint));
    imageFeature["timestamp"] = chrono::formatIsoDateTime(feature.timestamp());
    imageFeature["targetGeometry"] = geolib3::geojson(geolib3::mercator2GeoPoint(photo.shootingTarget));
    imageFeature["imageFull"] = [&](json::ObjectBuilder imageFull) {
        const auto size = mrc_common::transformByImageOrientation(
            feature.size(), feature.orientation());
        imageFull["url"] = makeFullImageUrl(mrcUrls, feature);
        imageFull["width"] = size.width;
        imageFull["height"] = size.height;
    };

    imageFeature["imagePreview"] = [&](json::ObjectBuilder imagePreview) {
        auto size = normThumbnailSizeOf(feature);
        imagePreview["url"] = makeImagePreviewUrl(mrcUrls, feature);
        imagePreview["width"] = size.width;
        imagePreview["height"] = size.height;
    };
}

std::string formatPermalink(PermalinkId permalinkId)
{
    const auto permalinkIdStr = std::to_string(permalinkId);
    return "[Altay " + permalinkIdStr + "](" + ALTAY_CARDS_URL + permalinkIdStr + ")";
}

geolib3::Point2
getBestFeedbackCoordinate(
    const std::map<PhotoId, mrc_db::Feature>& mrcFeatures,
    const CompanyDataBundle& bundle)
{
    for (const auto& photo : bundle.photos) {
        if (photo.subject == PhotoSubject::Facade && mrcFeatures.contains(photo.photoId)) {
            return photo.shootingTarget;
        }
    }
    for (const auto& photo : bundle.photos) {
        if (mrcFeatures.contains(photo.photoId)) {
            return photo.shootingTarget;
        }
    }
    return bundle.coordinate;
}

std::string
makeRequestBody(
    const std::map<PhotoId, mrc_db::Feature>& mrcFeatures,
    const CompanyDataBundle& bundle,
    const std::optional<editor::BasicEditorObject>& nmapsObject,
    const MrcUrls& mrcUrls)
{
    const auto privacy = evalMaxPrivacy(mrcFeatures);
    const auto position =
        nmapsObject
            ? geolib3::mercator2GeoPoint(getBestFeedbackCoordinate(mrcFeatures, bundle))
            : geolib3::mercator2GeoPoint(bundle.coordinate);
    const auto& taskSource = !nmapsObject
        ? CREATE_POI_SOURCE
        : (isVerifiedPoi(*nmapsObject)
            ? VERIFIED_POI_PHOTO_SOURCE
            : NOT_VERIFIED_POI_PHOTO_SOURCE);
    const auto action = nmapsObject
                ? fb::SuggestedAction::VerifyPosition
                : fb::SuggestedAction::CreatePoi;

    json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["source"] = taskSource;
        builder["position"] << geolib3::geojson(position);
        builder["suggestedAction"] = toString(action);
        builder["hidden"] = (mrc_db::FeaturePrivacy::Public != privacy);
        if (mrc_db::FeaturePrivacy::Secret == privacy) {
            builder["internalContent"] = true;
        }
        if (nmapsObject) {
            builder["objectId"] = std::to_string(nmapsObject->id);
        } else {
             builder["userComment"] = formatPermalink(bundle.permalinkId);
        }
        builder["sourceContext"] = [&](json::ObjectBuilder builder) {
            builder["type"] = "images";
            builder["content"] = [&](json::ObjectBuilder builder) {
                builder["imageFeatures"] = [&](json::ArrayBuilder builder) {
                    for (const auto& photo : bundle.photos) {
                        const auto it = mrcFeatures.find(photo.photoId);
                        if (it == mrcFeatures.end()) {
                            continue;
                        }
                        builder << [&](json::ObjectBuilder imageFeature) {
                            formatBundleFeature(imageFeature, photo, it->second, mrcUrls);
                        };
                    }
                };
            };
        };
    };
    return builder.str();
}

FeedbackTaskId
submitTask(const std::string& body, const Config& cfg)
{
    try {
        http::Client httpClient;
        http::Request request(httpClient, http::POST, cfg.socialBackofficeUrl() +
                FEEDBACK_TASKS_API_BASE + std::string{toString(fb::Type::Poi)});
        if (cfg.socialBackofficeTvmTicketProvider()) {
            request.addHeader(auth::SERVICE_TICKET_HEADER, cfg.socialBackofficeTvmTicketProvider()());
        }
        request.setContent(body);
        auto response = request.perform();
        REQUIRE(response.status() == HTTP_OK, "Feedback api error:" << response.status());
        const auto parsed = maps::json::Value::fromString(response.readBody());
        return std::stoll(parsed["feedbackTask"]["id"].toString());
    } catch (const std::exception& ex) {
        ERROR() << "Failed to create feedback task:" << ex.what() << "\n" << body;
        throw;
    }

}

FeedbackTaskId
createFeedbackTask(
    const std::map<PhotoId, mrc_db::Feature>& mrcFeatures,
    const CompanyDataBundle& bundle,
    const std::optional<editor::BasicEditorObject>& nmapsObject,
    const Config& cfg)
{
    const MrcUrls mrcUrls {
        .common = cfg.mrcBrowserUrl(),
        .pro = cfg.mrcBrowserProUrl()
    };
    const auto taskText = makeRequestBody(
        mrcFeatures,
        bundle,
        nmapsObject,
        mrcUrls);
    return submitTask(taskText, cfg);
}

std::optional<editor::BasicEditorObject>
getNmapsObject(const CompanyDataBundle& bundle, const Config& cfg)
{
    editor::Instance editor(cfg.editorUrl(), common::ROBOT_UID);
    const auto objects = editor.getObjectsByBusinessId(
        std::to_string(bundle.permalinkId),
        bundle.coordinate,
        PERMALINK_SEARCH_RADIUS);
    if (objects.empty()) {
        return {};
    }
    std::optional<editor::BasicEditorObject> closest;
    double minDistance = PERMALINK_SEARCH_RADIUS;
    for (const auto& object : objects) {
        const auto fullData = editor.getObjectById(object.id);
        const auto point = fullData.getGeometryInMercator()->get<geolib3::Point2>();
        const auto distance = mercatorCoordsDistaceMeters(bundle.coordinate, point);
        if (!closest || minDistance > distance) {
            closest = fullData;
            minDistance = distance;
        }
    }
    return closest;
}

bool
areAllPhotosProcessed(const std::map<PhotoId, mrc_db::Feature>& mrcFeatures)
{
    ASSERT(!mrcFeatures.empty());
    return std::all_of(
        mrcFeatures.begin(),
        mrcFeatures.end(),
        [](const auto& idFeature) {
            const auto& feature = idFeature.second;
            return
                feature.processedAt().has_value() &&
                feature.shouldBePublished() == feature.isPublished();
        });
}

} // namespace helpers

void
processQueue(const Config& cfg)
{
    CompanyDataBundleQueue queue(cfg.socialPool());
    auto bundles = queue.loadWaitingBatch(0, MAX_BUNDLES);
    for (auto& bundle : bundles) {
        auto nmapsObject = helpers::getNmapsObject(bundle, cfg);
        if (!nmapsObject) {
            continue; // disabled create tasks by NMAPS-15793
        }
        if (nmapsObject && !helpers::taskShouldBeCreated(bundle, *nmapsObject)) {
            queue.setBundleState(bundle.id, CompanyDataBundle::State::Canceled);
            continue;
        }
        auto mrcFeatures = helpers::loadBundleFeatures(bundle, cfg);
        if (!helpers::areAllPhotosProcessed(mrcFeatures)) {
            continue;
        }
        mrcFeatures = helpers::removeNotPublished(mrcFeatures);
        if (mrcFeatures.empty()) {
            queue.setBundleState(bundle.id, CompanyDataBundle::State::Canceled);
            continue;
        }
        try {
            const auto feedbackTaskId = helpers::createFeedbackTask(mrcFeatures, bundle, nmapsObject, cfg);
            queue.markAsSubmitted(bundle.id, feedbackTaskId);
        } catch (const std::exception& ex) {
            WARN() << "Failed to create task for bundle " << bundle.id << " exception: " << ex.what();
            queue.setBundleState(bundle.id, CompanyDataBundle::State::Failed);
        }
    }
}
} // namespace maps::wiki::walkers_export_downloader
