#include "storage.h"

#include <maps/libs/common/include/make_batches.h>

namespace maps::wiki::walkers_export_downloader {

namespace {

const size_t BATCH_SIZE = 100;

uint64_t
createVersion(const Config& cfg)
{
    auto socialTxn = cfg.socialPool().masterWriteableTransaction();
    const auto rows = socialTxn->exec(
        "INSERT INTO sprav.permalink_photo_versions (is_active) "
        "VALUES (false) RETURNING version_id");
    ASSERT(rows.size());
    socialTxn->commit();
    return rows[0][0].as<uint64_t>();
}

void
activateVersion(uint64_t versionId, const Config& cfg)
{
    auto socialTxn = cfg.socialPool().masterWriteableTransaction();
    socialTxn->exec(
        "UPDATE sprav.permalink_photo_versions "
        "SET is_active = (version_id = " +
        std::to_string(versionId) +")");
    socialTxn->commit();
}

void
deleteInactiveRecords(uint64_t versionId, const Config& cfg)
{
    auto socialTxn = cfg.socialPool().masterWriteableTransaction();
    socialTxn->exec(
        "DELETE FROM sprav.permalink_photo WHERE version_id <> " +
        std::to_string(versionId));
    socialTxn->commit();
}

void
storePhotoId(const PhotoId& photoId, const std::string& photoKey, const Config& cfg)
{
    auto socialTxn = cfg.socialPool().masterWriteableTransaction();
    socialTxn->exec("INSERT INTO sprav.photo_ids (photo_key, photo_id) "
        " VALUES(" + socialTxn->quote(photoKey) + ", " + socialTxn->quote(photoId) + ")"
        " ON CONFLICT DO NOTHING");
    socialTxn->commit();
}

} // namespace

Storage::Storage(const Config& cfg)
    : cfg_(cfg)
    , mrcClient_(cfg.mrcAgentProxyUrl())
{
    mrcClient_.setTvmTicketProvider(cfg.mrcAgentProxyTvmTicketProvider());
    auto socialTxn = cfg.socialPool().masterReadOnlyTransaction();

    auto photoRows = socialTxn->exec("SELECT photo_key, photo_id FROM sprav.photo_ids");
    photoIdsByKey_.reserve(photoRows.size());
    for (const auto& row : photoRows) {
        photoIdsByKey_.emplace(row[0].as<PhotoKey>(), row[1].as<PhotoId>());
    }

    auto comanyRows = socialTxn->exec("SELECT permalink_id, photo_id FROM sprav.permalink_photo");
    for (const auto& row : comanyRows) {
        photoIdsByPermalink_[row[0].as<PermalinkId>()].insert(row[1].as<PhotoId>());
    }
}

std::optional<PhotoId>
Storage::findPhoto(const PhotoKey& key) const
{
    const auto it = photoIdsByKey_.find(key);
    return it == photoIdsByKey_.end()
        ? std::optional<PhotoId>()
        : std::optional<PhotoId>(it->second);
}

void
Storage::assignToCompany(const PhotoId& photoId, PermalinkId permalinkId)
{
    photoIdsByPermalink_[permalinkId].insert(photoId);
}

void
Storage::writeNewVersionRows(uint64_t versionId) const
{
    auto socialTxn = cfg_.socialPool().masterWriteableTransaction();
    const auto batches = maps::common::makeBatches(photoIdsByPermalink_, BATCH_SIZE);
    const std::string insertStatement =
        "INSERT INTO sprav.permalink_photo (permalink_id, photo_id, version_id) "
        "VALUES ";
    for (const auto& batch : batches) {
        std::string valuesPart;
        for (const auto& [permalinkId, photoIds] : batch) {
            for (const auto& photoId : photoIds) {
                if (!valuesPart.empty()) {
                    valuesPart += ", ";
                }
                valuesPart +=
                    " (" +
                        std::to_string(permalinkId) + ", " +
                        socialTxn->quote(photoId) + ", " +
                        std::to_string(versionId) + ")";
            }
        }
        socialTxn->exec(insertStatement + valuesPart);
    }
    socialTxn->commit();
}

void
Storage::commitChanges() const
{
    const auto rowsVersionId = createVersion(cfg_);
    writeNewVersionRows(rowsVersionId);
    activateVersion(rowsVersionId, cfg_);
    deleteInactiveRecords(rowsVersionId, cfg_);
}

std::optional<PhotoId>
Storage::createObjectPhoto(
    const PhotoKey& key,
    const geolib3::Point2& coordinate,
    const geolib3::Point2& target,
    chrono::TimePoint takenAt,
    std::string image)
{
    REQUIRE(!findPhoto(key),
        "Photo already stored [" << key << "]");
    CreateObjectRequest request {
        .image = std::move(image),
        .coordinate = coordinate,
        .target = target,
        .takenAt = takenAt
    };
    const auto photoId = mrcClient_.submitObject(request);
    if (!photoId.empty()) {
        storePhotoId(photoId, key, cfg_);
        photoIdsByKey_.emplace(key, photoId);
        return photoId;
    }
    return {};
}

} // namespace maps::wiki::walkers_export_downloader
