#include "sync_database.h"

#include "common.h"
#include "company_data_bundle_queue.h"
#include "storage.h"
#include "yt_reader.h"
#include <maps/wikimap/mapspro/libs/http/http_utils.h>

#include <maps/libs/log8/include/log8.h>

using namespace std::literals::chrono_literals;

namespace maps::wiki::walkers_export_downloader {

namespace {

const auto MAX_PHOTO_AGE_FOR_BUNDLE = std::chrono::days(3);
const size_t DOWNLOAD_PHOTO_LIMIT = 100000;
const auto RETRY_POLICY = maps::common::RetryPolicy()
    .setTryNumber(5)
    .setInitialCooldown(1s)
    .setCooldownBackoff(2.0);

std::string readURLData(const std::string& dataUrl, http::Client& httpClient)
{
    http::URL url(dataUrl);
    auto [responseBody, status] = httpClient.get(url, RETRY_POLICY);
    if (status != http_status::OK) {
        WARN() << "Failed to make http get: " << url;
        return {};
    }
    return responseBody;
}

bool
storeWalkersPhoto(
    const WalkersExportData& companyData,
    const WalkersPhotoData& photoData,
    Storage& storage,
    http::Client& httpClient)
{
    const auto shootingTarget =
        photoData.coordObject
        ? photoData.coordObject
        : companyData.defaultShootingTarget;
    if (!shootingTarget) {
        return false;
    }
    bool isNewPhoto = false;
    auto photoId = storage.findPhoto(photoData.downloadUrl);
    if (!photoId) {
        photoId = storage.createObjectPhoto(
            photoData.downloadUrl,
            *photoData.coordTaking,
            *shootingTarget,
            photoData.submitTs,
            readURLData(photoData.downloadUrl, httpClient));
        isNewPhoto = true;
    }
    if (!photoId) {
        WARN()
            << "Failed to store photo: " <<  photoData.downloadUrl
            << " permalinkId: " << companyData.permalinkId;
        return false;
    }
    storage.assignToCompany(*photoId, companyData.permalinkId);
    return isNewPhoto;
}

void
mapMrcPhotoToPermalink(
    const WalkersYtData& walkersYtData,
    Storage& storage,
    std::map<PermalinkId, std::vector<WalkersPhotoData>>& newPhotosAdded)
{
    http::Client httpClient;
    const auto& walkersExport = walkersYtData.exportDatas;
    const auto& walkersPhotoDatas = walkersYtData.photoDatas;

    std::map<std::string, size_t> downloadUrlToCompany;
    for (size_t i = 0; i < walkersExport.size(); ++i) {
        const auto& companyData = walkersExport[i];
        for (const auto& photo : companyData.photos) {
            downloadUrlToCompany.emplace(photo, i);
        }
    }

    auto findCompany = [&](const WalkersPhotoData& photoData) {
        return !photoData.coordTaking
            ? downloadUrlToCompany.end()
            : downloadUrlToCompany.find(photoData.downloadUrl);
    };

    size_t total = 0;
    size_t notUsed = 0;
    size_t alreadyDownloaded = 0;
    for (const auto& photoData : walkersPhotoDatas) {
        if (findCompany(photoData) == downloadUrlToCompany.end()) {
            ++notUsed;
            continue;
        }
        if (storage.findPhoto(photoData.downloadUrl)) {
            ++alreadyDownloaded;
            continue;
        }
        ++total;
    }
    INFO() <<
        "Photo data incoming: " << walkersPhotoDatas.size() <<
        " Not referenced or task is already processed: " << notUsed <<
        " Already downloaded: " << alreadyDownloaded <<
        " Remaning: " << total;


    size_t added = 0;

    for (const auto& photoData : walkersPhotoDatas) {
        auto it = findCompany(photoData);
        if (it == downloadUrlToCompany.end()) {
            continue;
        }
        const auto& companyData = walkersExport[it->second];
        if (storeWalkersPhoto(companyData, photoData, storage, httpClient)) {
            ++added;
            newPhotosAdded[companyData.permalinkId].push_back(photoData);
            if (added % 10 == 0) {
                INFO() << "Downloaded photos: " << added << "/" << total;
            }
        }
        if (added >= DOWNLOAD_PHOTO_LIMIT) {
            INFO() << "Limit " << DOWNLOAD_PHOTO_LIMIT << " reached";
            break;
        }
    }
    INFO() << "Total downloaded photos " << added;

    for (const auto& [downloadUrl, companyDataIdx] : downloadUrlToCompany) {
        auto photoId = storage.findPhoto(downloadUrl);
        if (photoId) {
            storage.assignToCompany(*photoId, walkersExport[companyDataIdx].permalinkId);
        }
    }
}

void syncWalkersData(
    Storage& storage,
    ReaderContext& readerContext,
    std::map<PermalinkId, std::vector<WalkersPhotoData>>& newPhotosAdded)
{
    INFO() << "syncWalkersData started.";
    const auto walkersYtData = readWalkersYtData(readerContext);
    REQUIRE(!walkersYtData.exportDatas.empty() && !walkersYtData.photoDatas.empty(),
        "Failed to get walkers data from YT.");
    mapMrcPhotoToPermalink(walkersYtData, storage, newPhotosAdded);
    INFO() << "syncWalkersData complete.";
}

void syncRawDesktopWalkersData(
    Storage& storage,
    ReaderContext& readerContext,
    std::map<PermalinkId, std::vector<WalkersPhotoData>>& newPhotosAdded)
{
    INFO() << "syncRawDesktopWalkersData started.";
    const auto walkersYtData = readRawDesktopWalkersYtData(readerContext);
    REQUIRE(!walkersYtData.exportDatas.empty() && !walkersYtData.photoDatas.empty(),
        "Failed to get raw desktop walkers data from YT.");
    mapMrcPhotoToPermalink(walkersYtData, storage, newPhotosAdded);
    INFO() << "syncRawDesktopWalkersData complete.";
}

void enqueueNewData(
    const Config& cfg,
    const Storage& storage,
    const ReaderContext& readerContext,
    const std::map<PermalinkId, std::vector<WalkersPhotoData>>& newPhotosAdded)
{
    CompanyDataBundleQueue queue(cfg.socialPool());
    const auto now = maps::chrono::TimePoint::clock::now();
    for (const auto& [permalinkId, photos] : newPhotosAdded) {
        const auto poiCoord = readerContext.poiCoordinate(permalinkId);
        if (!poiCoord) {
            WARN() << "CompanyDataBundleQueue: SKIP Can't get coords for: " << permalinkId;
            continue;
        }
        if (photos.empty() ||
            std::all_of(photos.begin(), photos.end(), [&now](const auto& photo) {
                return now - photo.submitTs > MAX_PHOTO_AGE_FOR_BUNDLE;
            }))
        {
            continue;
        };
        const auto existingBudleIds = queue.findBundles(permalinkId, CompanyDataBundle::State::Waiting);
        BundleId bundleId =
            existingBudleIds.size() == 1
            ? existingBudleIds.front()
            : 0;
        if (bundleId) {
            queue.setBundleState(bundleId, CompanyDataBundle::State::Draft);
        } else {
            bundleId = queue.createBundle(permalinkId, *poiCoord);
        }
        for (const auto& photo : photos) {
            const auto photoId = storage.findPhoto(photo.downloadUrl);
            auto shootingTarget = photo.coordObject;
            if (!shootingTarget) {
                WARN() << "Using poi coord as shootingTarget, permalinkId: " << permalinkId << " photoUrl: " << photo.downloadUrl;
                shootingTarget = poiCoord;
            }
            ASSERT(photoId);
            ASSERT(photo.coordTaking && shootingTarget);
            queue.storePhotoData(PhotoData {
                .photoId = *photoId,
                .subject = photo.subject,
                .shootingPoint = *photo.coordTaking,
                .shootingTarget = *shootingTarget
            });
            queue.assignPhotoToBundle(bundleId, *photoId);
        }
        queue.setBundleState(bundleId, CompanyDataBundle::State::Waiting);
    }
}

} // namespace

void syncDatabase(const Config& cfg)
{
    Storage storage(cfg);
    ReaderContext readerContext;
    readerContext.poiCoordinates = readAltayPOICoordinates();
    std::map<PermalinkId, std::vector<WalkersPhotoData>> newPhotosAdded;
    syncRawDesktopWalkersData(storage, readerContext, newPhotosAdded);
    syncWalkersData(storage, readerContext, newPhotosAdded);
    storage.commitChanges();
    enqueueNewData(cfg, storage, readerContext, newPhotosAdded);
}

} // namespace maps::wiki::walkers_export_downloader
