#include "yt_reader.h"

#include <maps/wikimap/mapspro/libs/poi_feed/include/feed_object_data.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/yt.h>
#include <maps/libs/geolib/include/conversion.h>

#include <sprav/protos/export.pb.h>
#include <mapreduce/yt/interface/client.h>

#include <algorithm>
#include <cctype>
#include <string>

using namespace NYT;

namespace maps::wiki::walkers_export_downloader {

namespace {

const TString YT_REALM = "hahn";
const TString WALKERS_EXPORT_URL = "//home/sprav/assay/tasker/walkers/export";
const TString WALKERS_MOBILE_PHOTOS = "//home/sprav/assay/tasker/walkers/v3/yang_mobile_photos";
const TString WALKERS_DESKTOP_PHOTOS = "//home/sprav/assay/tasker/walkers/v3/yang_desktop_photos";
const TString WALKERS_DESKTOP_RAW = "//home/sprav/assay/tasker/walkers/v3/yang_raw_desktop";
const TString ALTAY_POI_EXPORT = "//home/altay/db/export/current-state/exported/company";

const std::string MAIN_ENTRANCE_TYPE = "main";

const std::map<std::string, PhotoSubject> RAW_TABLE_ACCEPTED_OBJECT_TAGS_TO_SUBJECT {
    {MAIN_ENTRANCE_TYPE, PhotoSubject::Entrance},
    {"entrance", PhotoSubject::Entrance},
    {"facade", PhotoSubject::Facade},
};

const auto MAX_PHOTO_AGE = std::chrono::years(1);

struct WalkersExportStat
{
    size_t noPermalink = 0;
    size_t noPhoto = 0;
    size_t noActualization = 0;
    size_t emptyPhotos = 0;
    size_t photosNotParsed = 0;
};

std::string toLower(std::string entranceType)
{
    std::transform(
        entranceType.begin(),
        entranceType.end(),
        entranceType.begin(),
        [](unsigned char c) { return std::tolower(c); });
    return entranceType;
}

std::string normalizeRawTableTag(std::string tag)
{
    tag = toLower(tag);
    if (tag == MAIN_ENTRANCE_TYPE) {
        return "entrance";
    }
    return tag;
}

bool
isMainEntranceType(std::string entranceType)
{
    return toLower(std::move(entranceType)) == MAIN_ENTRANCE_TYPE;
}


bool
hasNonZeroDigits(const std::string& coordStr)
{
    return std::any_of(coordStr.begin(), coordStr.end(),
        [](const auto& c) {
            return c != '0' && std::isdigit(c);
        });
}

// Parse coordinate in form of 59° 54' 13.22"
double
parseDegreeCoord(const std::string& coordStr)
{
    double vals[3] = {0,0,0};
    size_t curVal = 0;
    double ref = 1.0;
    std::string curStr;
    for (const auto& c : coordStr) {
        if (c == '-') {
            ref = -1.0;
        } else if (std::isdigit(c) || c == '.') {
            curStr += c;
        } else if (!curStr.empty()) {
            vals[curVal] = std::atof(curStr.c_str());
            if (curVal == 2) {
                break;
            }
            ++curVal;
            curStr.clear();
        }
    }
    return ref * (vals[0] + vals[1] / 60 + vals[2] / 3600);
}

void
readTablePhotos(
    NYT::IClientPtr& client,
    const TString& tableName,
    const size_t limit,
    maps::chrono::TimePoint now,
    std::map<std::string, WalkersPhotoData>& result)
{
    auto reader = client->CreateTableReader<NYT::TNode>(tableName);
    for (; (!limit || result.size() < limit) && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        if (!row["downloadUrl"].IsString()) {
            continue;
        }
        const auto& downloadUrl = row["downloadUrl"].AsString();
        if (result.contains(downloadUrl)) {
            continue;
        }
        if (!row["tag"].IsString() || row["tag"].AsString() != "facade") {
            continue;
        }
        WalkersPhotoData walkersPhotoData;
        walkersPhotoData.downloadUrl = downloadUrl;
        walkersPhotoData.submitTs =
            chrono::sinceEpochToTimePoint<std::chrono::milliseconds>(
                row["submitTs"].IntCast<uint64_t>());
        if (now - walkersPhotoData.submitTs > MAX_PHOTO_AGE) {
            continue;
        }
        if (row["exif"].IsNull()) {
            continue;
        }
        const auto& exif = row["exif"].AsMap();
        if (!exif.contains("gps") || exif.at("gps").IsNull()) {
            continue;
        }
        const auto& gps = exif.at("gps").AsMap();
        if (!gps.contains("gps_latitude") || !gps.contains("gps_longitude")) {
            continue;
        }
        const auto lon = gps.at("gps_longitude").AsString();
        const auto lat = gps.at("gps_latitude").AsString();
        if (!hasNonZeroDigits(lon) && !hasNonZeroDigits(lat)) {
            continue;
        }

        walkersPhotoData.coordTaking = geolib3::geoPoint2Mercator(geolib3::Point2(
            parseDegreeCoord(lon),
            parseDegreeCoord(lat)));
        result[downloadUrl] = std::move(walkersPhotoData);
    }
}

struct ShootingInfo
{
    geolib3::Point2 taking;
    geolib3::Point2 target;
};

std::optional<ShootingInfo>
findShootingPoints(
    const std::map<std::string, geolib3::Point2>& tagToCoord,
    const std::string& tag)
{
    std::string finalTag = tag;
    if (!tagToCoord.contains(finalTag)) {
        if (finalTag == "facade") {
            finalTag = "entrance";
        } else {
            return {};
        }
    }

    const auto tagTaking = finalTag + "_taking";
    auto targetIt = tagToCoord.find(finalTag);
    auto takingIt = tagToCoord.find(tagTaking);
    if (targetIt == tagToCoord.end() || takingIt == tagToCoord.end()) {
        return {};
    }
    return ShootingInfo {
        .taking = takingIt->second,
        .target = targetIt->second
    };
}

bool
readTableRawPhotosRow(
    const NYT::TNode& row,
    maps::chrono::TimePoint now,
    std::map<std::string, WalkersPhotoData>& result)
{
    const auto& json = row["json"].AsMap();
    const auto submitTs =
        chrono::sinceEpochToTimePoint<std::chrono::milliseconds>(
            row["submitTs"].IntCast<uint64_t>());
    if (now - submitTs > MAX_PHOTO_AGE) {
        return false;
    }

    if (!json.contains("photos") || !json.contains("entrances")) {
        return false;
    }
    const auto& photos = json.at("photos").AsList();
    const auto& entrances = json.at("entrances").AsList();

    std::map<std::string, std::string> urlToTag;
    std::map<std::string, geolib3::Point2> tagToCoord;

    auto hasTakingTag = false;
    for (const auto& entrance : entrances) {
        const auto& jsonEntrance = entrance.AsMap();
        const std::string tag = normalizeRawTableTag(jsonEntrance.at("type").AsString());
        if (tag.ends_with("_taking")) {
            hasTakingTag = true;
        }
        tagToCoord[tag] =
            geolib3::geoPoint2Mercator(geolib3::Point2{
                jsonEntrance.at("lon").ConvertTo<double>(),
                jsonEntrance.at("lat").ConvertTo<double>(),
            });
    }
    if (!hasTakingTag) {
        return false;
    }

    for (const auto& photo : photos) {
        const auto& jsonPhoto = photo.AsMap();
        const auto url = jsonPhoto.at("url").AsString();
        if (result.contains(url)) {
            return true;
        }
        auto photoTag = normalizeRawTableTag(jsonPhoto.at("tag").AsString());
        urlToTag[jsonPhoto.at("url").AsString()] = photoTag;
    }
    bool arePhotosAdded = false;
    for (const auto& [url, tag] : urlToTag) {
        const auto it = RAW_TABLE_ACCEPTED_OBJECT_TAGS_TO_SUBJECT.find(tag);
        if (it == RAW_TABLE_ACCEPTED_OBJECT_TAGS_TO_SUBJECT.end()) {
            continue;
        }
        const auto shootingPoints = findShootingPoints(tagToCoord, tag);
        if (shootingPoints) {
            WalkersPhotoData walkersPhotoData;
            walkersPhotoData.downloadUrl = url;
            walkersPhotoData.submitTs = submitTs;
            walkersPhotoData.coordTaking = shootingPoints->taking;
            walkersPhotoData.coordObject = shootingPoints->target;
            walkersPhotoData.subject = it->second;
            result[url] = std::move(walkersPhotoData);
            arePhotosAdded = true;
        }
    }
    return arePhotosAdded;
}


std::optional<WalkersExportData>
readCompanyExportRow(
    const NYT::TNode& row,
    WalkersExportStat& stat)
{
    const auto id = row["id"].IntCast<uint64_t>();
    const auto& json = row["json"].AsMap();
    if (!json.contains("permalink")) {
        ++stat.noPermalink;
        return {};
    }
    if (!json.contains("photos")) {
        ++stat.noPhoto;
        return {};
    }
    if (!json.contains("actualization-date")) {
        ++stat.noActualization;
        return {};
    }
    const auto& photos = json.at("photos").AsList();
    if (photos.empty()) {
        ++stat.emptyPhotos;
        return {};
    }
    WalkersExportData data {
        .permalinkId = json.at("permalink").IntCast<poi_feed::PermalinkId>(),
        .actualizationDate =
            chrono::sinceEpochToTimePoint<std::chrono::milliseconds>
                (json.at("actualization-date").IntCast<int64_t>())
    };
    for (const auto& photo : photos) {
        if (photo.IsMap()) {
            const auto& photoMap = photo.AsMap();
            if (!photoMap.contains("url") || !photoMap.at("url").IsString()) {
                WARN() << "Failed to parse photos Array of maps: " << id;
                ++stat.photosNotParsed;
                return {};
            }
            data.photos.emplace_back(photoMap.at("url").AsString());
        } else if (photo.IsList()) {
            const auto& subPhotos = photo.AsList();
            for (const auto& subPhoto : subPhotos) {
                if (!subPhoto.IsString()) {
                    WARN() << "Failed to parse photos Array of array: " << id;
                    ++stat.photosNotParsed;
                    return {};
                }
                data.photos.emplace_back(subPhoto.AsString());
            }
        } else if (photo.IsString()) {
            data.photos.emplace_back(photo.AsString());
        } else {
            WARN() << "Failed to parse photos Array of string: " << id;
            ++stat.photosNotParsed;
        }
    }
    const auto entrancesIt = json.find("entrances");
    if (entrancesIt != json.end()) {
        const auto& entrances = entrancesIt->second.AsList();
        for (const auto& entrance : entrances) {
            const auto& entranceMap = entrance.AsMap();
            const auto entranceType = entranceMap.at("type").AsString();
            if (!data.defaultShootingTarget || isMainEntranceType(entranceType)) {
                data.defaultShootingTarget = geolib3::geoPoint2Mercator(
                    geolib3::Point2 {
                        entranceMap.at("lon").ConvertTo<double>(),
                        entranceMap.at("lat").ConvertTo<double>(),
                });
            }
        }
    }
    return data;
}

void
readTableRaw(
    NYT::IClientPtr& client,
    const TString& tableName,
    const size_t limit,
    maps::chrono::TimePoint now,
    std::map<std::string, WalkersPhotoData>& photos,
    WalkersExportDatas& companies,
    WalkersExportStat& stat,
    ReaderContext& readerContext)
{
    auto reader = client->CreateTableReader<NYT::TNode>(tableName);
    for (; (!limit || photos.size() < limit) && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        const auto walkerTaskId = row["id"].IntCast<uint64_t>();
        if (readerContext.seenWalkersTasksIds.contains(walkerTaskId)) {
            continue;
        }
        readerContext.seenWalkersTasksIds.insert(walkerTaskId);
        if (readTableRawPhotosRow(row, now, photos)) {
            auto data = readCompanyExportRow(row, stat);
            if (!data) {
                continue;
            }
            companies.emplace_back(std::move(*data));
        }
    }
}

void
readTableExport(
    NYT::IClientPtr& client,
    const TString& tableName,
    const size_t limit,
    WalkersExportDatas& result,
    WalkersExportStat& stat,
    ReaderContext& readerContext)
{
    auto reader = client->CreateTableReader<NYT::TNode>(tableName);
    for (size_t i = 0; (!limit || i < limit) && reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        const auto walkerTaskId = row["id"].IntCast<uint64_t>();
        if (readerContext.seenWalkersTasksIds.contains(walkerTaskId)) {
            continue;
        }
        readerContext.seenWalkersTasksIds.insert(walkerTaskId);
        auto data = readCompanyExportRow(row, stat);
        if (!data) {
            continue;
        }
        ++i;
        result.emplace_back(std::move(*data));
    }
}

void
addMissingDefaultShootingTargets(WalkersExportDatas& companies, const ReaderContext& readerContext)
{
    bool hasMissing = false;
    for (const auto& company : companies) {
        if (!company.defaultShootingTarget) {
            hasMissing = true;
            break;
        }
    }
    if (!hasMissing) {
        return;
    }
    size_t updated = 0;
    for (auto& companyData : companies) {
        if (companyData.defaultShootingTarget) {
            continue;
        }
        const auto poiCoord = readerContext.poiCoordinate(companyData.permalinkId);
        if (!poiCoord) {
            continue;
        }
        companyData.defaultShootingTarget = *poiCoord;
        ++updated;
    }
    INFO() << "Missing defaultShootingTargets read: " << updated;
}

void
removeCompaniesWithoutShootingTarget(WalkersExportDatas& companies)
{
    size_t removed = 0;
    for (auto it = companies.begin(); it != companies.end(); ) {
        if (!it->defaultShootingTarget) {
            it = companies.erase(it);
            ++removed;
        } else {
            ++it;
        }
    }
    INFO() << "POIs without defaultShootingTarget removed: " << removed;
}

} // namespace

WalkersYtData
readWalkersYtData(ReaderContext& readerContext, const size_t limit)
{
    NYT::JoblessInitialize();
    auto client = common::yt::createYtClient(YT_REALM);

    WalkersYtData result;
    WalkersExportStat stat;
    const auto now = maps::chrono::TimePoint::clock::now();

    std::map<std::string, WalkersPhotoData> photoDatasByUrl;

    readTablePhotos(client, WALKERS_MOBILE_PHOTOS, limit, now, photoDatasByUrl);
    readTablePhotos(client, WALKERS_DESKTOP_PHOTOS, limit, now, photoDatasByUrl);
    readTableExport(client, WALKERS_EXPORT_URL, limit, result.exportDatas, stat, readerContext);
    addMissingDefaultShootingTargets(result.exportDatas, readerContext);
    removeCompaniesWithoutShootingTarget(result.exportDatas);
    for (const auto& [_, photoData] : photoDatasByUrl) {
        result.photoDatas.push_back(photoData);
    }
    std::sort(result.photoDatas.begin(), result.photoDatas.end(),
        [&](const auto& l, const auto& r) {
            return l.submitTs > r.submitTs;
        });

    INFO() << "Records read: " << result.exportDatas.size();
    INFO()
        << " No permalink: " << stat.noPermalink
        << " No actualization-date: " << stat.noActualization
        << " No photos field: " << stat.noPhoto
        << " Photo empty: " << stat.emptyPhotos
        << " Photos not parsed: " << stat.photosNotParsed;

    return result;
}

WalkersYtData
readRawDesktopWalkersYtData(ReaderContext& readerContext, size_t limit)
{
    NYT::JoblessInitialize();
    auto client = common::yt::createYtClient(YT_REALM);

    WalkersYtData result;
    WalkersExportStat stat;
    const auto now = maps::chrono::TimePoint::clock::now();

    std::map<std::string, WalkersPhotoData> photoDatasByUrl;
    readTableRaw(
        client,
        WALKERS_DESKTOP_RAW,
        limit,
        now,
        photoDatasByUrl,
        result.exportDatas,
        stat,
        readerContext);

    for (const auto& [_, photoData] : photoDatasByUrl) {
        result.photoDatas.push_back(photoData);
    }

    std::sort(result.photoDatas.begin(), result.photoDatas.end(),
        [&](const auto& l, const auto& r) {
            return l.submitTs > r.submitTs;
        });

    INFO() << "RAW Records read: " << result.exportDatas.size();
    INFO()
        << " No permalink: " << stat.noPermalink
        << " No actualization-date: " << stat.noActualization
        << " No photos field: " << stat.noPhoto
        << " Photo empty: " << stat.emptyPhotos
        << " Photos not parsed: " << stat.photosNotParsed;

    return result;
}

POICoords
readAltayPOICoordinates()
{
    NYT::JoblessInitialize();
    auto client = common::yt::createYtClient(YT_REALM);
    POICoords result;
    for (auto reader = client->CreateTableReader<NYT::TNode>(ALTAY_POI_EXPORT); reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        NSpravExport::TExportedCompany exportProto;
        Y_PROTOBUF_SUPPRESS_NODISCARD exportProto.ParseFromString(row["exported_company"].AsString());
        if (exportProto.HasGeo()) {
            const auto& pos = exportProto.GetGeo().GetLocation().GetPos();
            result.emplace(
                row["permalink"].IntCast<poi_feed::PermalinkId>(),
                geolib3::geoPoint2Mercator(geolib3::Point2{
                    pos.GetLon(),
                    pos.GetLat()}));
        }
    }
    return result;
}

std::optional<geolib3::Point2>
ReaderContext::poiCoordinate(PermalinkId permalinkId) const
{
    const auto it = poiCoordinates.find(permalinkId);
    return it == poiCoordinates.end()
        ? std::optional<geolib3::Point2>{}
        : it->second;
}
} // namespace maps::wiki::walkers_export_downloader
