#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <yandex/maps/wiki/unittest/arcadia.h>
#include <yandex/maps/wiki/unittest/json_schema.h>

#include <maps/wikimap/mapspro/services/tasks_sprav/src/walkers_export_downloader/lib/company_data_bundle_queue.h>
#include <maps/wikimap/mapspro/services/tasks_sprav/src/walkers_export_downloader/lib/process_queue_helpers.h>

#include <maps/wikimap/mapspro/services/mrc/libs/db/include/feature_gateway.h>
#include <yandex/maps/mrc/unittest/database_fixture.h>

using namespace maps::wiki;
using namespace maps::wiki::walkers_export_downloader;
namespace mrc_unittest = maps::mrc::unittest;

namespace maps::wiki::walkers_export_downloader::tests {

namespace {

const PermalinkId PERMALINK = 10;
const PhotoId PHOTO_ID = "10";
const geolib3::Point2 SHOOTING_POINT(4405371.443589358, 6877399.153418001);
const geolib3::Point2 SHOOTING_TARGET(4405371.443589358, 6877399.153418001);
const geolib3::Point2 POI_COORD(4405371.443589358, 6877399.153418001);
const PhotoData PHOTO_DATA {
        PHOTO_ID,
        PhotoSubject::Entrance,
        SHOOTING_POINT,
        SHOOTING_TARGET
    };
const FeedbackTaskId TEST_TASK_ID = 111;
const std::string TEST_UID = "1";
const std::string DEVICE_ID = "MyFluffyPhone";
const std::string COMMENT = "Hello, world!";
const std::string MDS_KEY = "4436";
const std::string MDS_PATH = "development/walk/images/17673352";
const helpers::MrcUrls MRC_URLS {
    "http://common",
    "http://pro",
};


const std::string SOCIAL_FEEDBACK_SCHEMA_PATH =
    ArcadiaSourceRoot() + "/maps/wikimap/mapspro/schemas/social/social.post.poi_feedback_task.request.schema.json";

enum class Publish { Yes, No };

class MrcDbFixture : public mrc_unittest::WithUnittestConfig<mrc_unittest::DatabaseFixture>
{
public:
    static MrcDbFixture& instance()
    {
        static MrcDbFixture fixture;
        return fixture;
    }


    PhotoId addPhoto(
        std::optional<Publish> publish,
        mrc_db::FeaturePrivacy privacy = mrc_db::FeaturePrivacy::Public)
    {
        auto txn = pool().masterWriteableTransaction();
        auto feature = sql_chemistry::GatewayAccess<mrc_db::Feature>::construct()
            .setGeodeticPos(geolib3::Point2(0.0, 0.0))
            .setHeading(geolib3::Heading(0.0))
            .setTimestamp(maps::chrono::parseIsoDate("2021-06-04 16:35:15"))
            .setMdsKey({MDS_KEY, MDS_PATH})
            .setSize(64, 64)
            .setUserId(TEST_UID)
            .setPrivacy(privacy);
        if (publish.has_value()) {
            feature
                .setAutomaticShouldBePublished(publish.value() == Publish::Yes)
                .setIsPublished(publish.value() == Publish::Yes)
                .setProcessedAt(maps::chrono::parseIsoDate("2021-06-04 16:54:57"));
        }
        mrc_db::FeatureGateway{*txn}.insert(feature);
        txn->commit();
        return std::to_string(feature.id());
    }

    std::map<PhotoId, mrc_db::Feature>
    loadPhotos(const std::set<PhotoId>& ids)
    {
        maps::mrc::db::TIdSet featureIds;
        auto txn = pool().masterWriteableTransaction();
        for (const auto& id : ids) {
            featureIds.insert(std::stoll(id));
        }
        auto mrcFeatures = mrc_db::FeatureGateway(*txn).loadByIds(
                {featureIds.begin(), featureIds.end()});
        txn->commit();
        std::map<PhotoId, mrc_db::Feature> result;
        for (const auto& feature : mrcFeatures) {
            result.emplace(std::to_string(feature.id()), feature);
        }
        return result;
    }
};



} // namespace

Y_UNIT_TEST_SUITE(photo_data_queue)
{

Y_UNIT_TEST_F(createBundle, unittest::ArcadiaDbFixture)
{
    CompanyDataBundleQueue queue(pool());
    const auto bundleId = queue.createBundle(PERMALINK, POI_COORD);
    UNIT_ASSERT(bundleId);
    UNIT_ASSERT_EQUAL(queue.waitingCount(), 0);
    const auto bundles = queue.load({bundleId});
    UNIT_ASSERT_EQUAL(bundles.size(), 1);
    UNIT_ASSERT_EQUAL(bundles[0].id, bundleId);
}

Y_UNIT_TEST_F(storePhotoData, unittest::ArcadiaDbFixture)
{
    CompanyDataBundleQueue queue(pool());

    queue.storePhotoData(PHOTO_DATA);
    UNIT_ASSERT(!queue.loadPhotoData("bebe"));
    const auto storedData = queue.loadPhotoData(PHOTO_ID);
    UNIT_ASSERT(storedData && *storedData == PHOTO_DATA);
}

Y_UNIT_TEST_F(fillBundle, unittest::ArcadiaDbFixture)
{
    CompanyDataBundleQueue queue(pool());
    queue.createBundle(PERMALINK, POI_COORD);
    const auto bundlesIds = queue.findBundles(PERMALINK, CompanyDataBundle::State::Draft);
    UNIT_ASSERT(!bundlesIds.empty());
    const auto bundleId = bundlesIds[0];
    queue.storePhotoData(PHOTO_DATA);
    queue.assignPhotoToBundle(bundleId, PHOTO_ID);
    queue.setBundleState(bundleId, CompanyDataBundle::State::Waiting);
    UNIT_ASSERT_EQUAL(queue.waitingCount(), 1);
    const auto bundles = queue.loadWaitingBatch(0, 0);
    UNIT_ASSERT_EQUAL(bundles.size(), 1);
    const auto& bundle = bundles[0];
    UNIT_ASSERT(!bundle.photos.empty() && bundle.photos[0] == PHOTO_DATA);
    queue.markAsSubmitted(bundleId, TEST_TASK_ID);
    const auto submitted = queue.load({bundleId})[0];
    UNIT_ASSERT(submitted.submittedAt);
    UNIT_ASSERT(submitted.feedbackTaskId == TEST_TASK_ID);
}

Y_UNIT_TEST_F(notProcessedPhotosPresent, unittest::ArcadiaDbFixture)
{
    std::set<PhotoId> photoIds;
    photoIds.insert(MrcDbFixture::instance().addPhoto(Publish::No));
    photoIds.insert(MrcDbFixture::instance().addPhoto(Publish::Yes));
    photoIds.insert(MrcDbFixture::instance().addPhoto(std::nullopt));
    auto mrcFeatures = MrcDbFixture::instance().loadPhotos(photoIds);
    UNIT_ASSERT(mrcFeatures.size() == 3);
    UNIT_ASSERT(!helpers::areAllPhotosProcessed(mrcFeatures));
}

Y_UNIT_TEST_F(createFeedbackTask, unittest::ArcadiaDbFixture)
{
    std::set<PhotoId> photoIds;
    photoIds.insert(MrcDbFixture::instance().addPhoto(Publish::No));
    photoIds.insert(MrcDbFixture::instance().addPhoto(Publish::Yes));
    photoIds.insert(MrcDbFixture::instance().addPhoto(Publish::Yes));
    auto mrcFeatures = MrcDbFixture::instance().loadPhotos(photoIds);
    UNIT_ASSERT(mrcFeatures.size() == 3);
    UNIT_ASSERT(helpers::areAllPhotosProcessed(mrcFeatures));
    mrcFeatures = helpers::removeNotPublished(mrcFeatures);
    UNIT_ASSERT(mrcFeatures.size() == 2);
    CompanyDataBundleQueue queue(pool());
    for (const auto& photoId : photoIds) {
        PhotoData data = PHOTO_DATA;
        data.photoId = photoId;
        queue.storePhotoData(data);
    }
    const auto bundleId = queue.createBundle(PERMALINK, POI_COORD);
    for (const auto& photoId : photoIds) {
        queue.assignPhotoToBundle(bundleId, photoId);
    }
    queue.setBundleState(bundleId, CompanyDataBundle::State::Waiting);
    const auto bundle = queue.load({bundleId})[0];
    const std::string taskRequest = makeRequestBody(
        mrcFeatures,
        bundle,
        std::nullopt,
        MRC_URLS);
     wiki::unittest::validateJson(taskRequest, SOCIAL_FEEDBACK_SCHEMA_PATH);
}

} // Y_UNIT_TEST_SUITE(photo_data_queue)

} // namespace maps::wiki::walkers_export_downloader::tests
