#include "client.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/retry_duration.h>
#include <util/string/cast.h>

using namespace std::string_literals;
using namespace maps::mrc::toloka;

namespace maps::wiki::yang {

namespace {

const auto API_HOST = "yang.yandex-team.ru"s;

std::string toColumnName(std::string key)
{
    std::transform(
        key.begin(),
        key.end(),
        key.begin(),
        [](unsigned char c) -> unsigned char {
            return c == ':' || c == '-' ? '_' : std::tolower(c);
        }
    );
    return key;
}

std::string timePointToStr(maps::chrono::TimePoint timePoint)
{
    auto str = maps::chrono::formatIsoDateTime(timePoint);
    return str.substr(0, str.find('Z'));
}

yang::Assignment makeAssignment(PoolId poolId, const io::Assignment& assignment)
{
    yang::Assignment a;

    a.add("assignment_link",
            "https://" + API_HOST + "/task/" + std::to_string(poolId) + "/" + assignment.id());
    a.add("assignment_assignment_id", assignment.id());
    a.add("assignment_worker_id", assignment.userId());
    a.add("assignment_status", "APPROVED");
    a.add("assignment_started", timePointToStr(assignment.createdAt()));

    const auto& tasks = assignment.tasks();
    ASSERT(tasks.size() == 1);
    const auto& task = tasks.front();
    a.add("assignment_task_id", task.id());
    const auto& inputValues = task.inputValues();
    ASSERT(inputValues.isObject());

    for (const auto& field : inputValues.fields()) {
        const auto& value = inputValues[field];
        a.add(
            toColumnName("input_" + field),
            value.isString()
                ? value.toString()
                : (std::ostringstream() << value).str());
    }

    const auto& solutions = assignment.solutions();
    ASSERT(solutions.size() == 1);
    const auto& outputValues = solutions.front().outputValues();
    ASSERT(outputValues.isObject());

    for (const auto& field : outputValues.fields()) {
        const auto& value = outputValues[field];
        if (value.isNull()) {
            WARN() << "Skip NULL value for field: " << field;
            continue;
        }
        auto column = toColumnName("output_" + field);
        if (value.isString()) {
            a.add(column, value.toString());
        } else {
            REQUIRE(field == "object-photos", "Unexpected non-string key: " << field);
            ASSERT(value.isArray());
            std::string str;
            for (const auto& v : value) {
                ASSERT(v.isString());
                str += (str.empty() ? "" : ",") + v.toString();
            }
            a.add(column, std::move(str));
        }
    }

    return a;
}

auto loadAcceptedAssignments(
    const io::TolokaClient& client,
    ProjectId projectId,
    const std::string& poolId)
{
    io::Filter filter;
    filter.byProjectId(std::to_string(projectId));
    filter.byPoolId(poolId);
    filter.byAssignmentStatus(io::AssignmentStatus::Accepted);
    return common::retryDuration([&] {
        return client.getAssignments(filter);
    });
}

auto loadClosedPools(
    const io::TolokaClient& client,
    ProjectId projectId)
{
    io::Filter filter;
    filter.byProjectId(std::to_string(projectId));
    filter.byPoolStatus(io::PoolStatus::Closed);

    return common::retryDuration([&] {
        return client.getPools(filter);
    });
}

auto loadPool(
    const io::TolokaClient& client,
    PoolId poolId)
{
    return common::retryDuration([&] {
        return client.getPool(std::to_string(poolId));
    });
}

auto loadAttachment(
    const io::TolokaClient& client,
    const AttachmentId& attachmentId)
{
    return common::retryDuration([&] {
        return client.getAttachment(attachmentId);
    });
}

} // namespace

Client::Client(const std::string& oauthToken, ProjectId projectId)
    : client_(API_HOST, "OAuth " + oauthToken)
    , projectId_(projectId)
{
    REQUIRE(!oauthToken.empty(), "Empty YANG_TOKEN");
}

std::deque<PoolId> Client::getClosedPoolIds() const
{
    auto response = loadClosedPools(client_, projectId_);

    std::deque<PoolId> result;
    for (const auto& pool : response.pools()) {
        PoolId poolId = 0;
        REQUIRE(TryFromString(pool.id(), poolId),
                "Non-numeric pool id : " << pool.id());
        result.push_back(poolId);
    }
    return result;
}

void Client::listClosedPools() const
{
    auto response = loadClosedPools(client_, projectId_);

    struct PoolData {
        PoolData(io::Pool pool, size_t size)
            : pool(std::move(pool))
            , size(size)
        {}

        bool operator < (const PoolData& p) const
        {
            return pool.privateName() < p.pool.privateName();
        }

        io::Pool pool;
        size_t size;
    };

    std::vector<PoolData> pools;
    for (const auto& pool : response.pools()) {
        auto response = loadAcceptedAssignments(client_, projectId_, pool.id());
        INFO() << "Analyzing pool " << pool.id() << " : " << pool.privateName();
        pools.emplace_back(pool, response.assignments().size());
    }

    std::sort(pools.begin(), pools.end());
    for (const auto& [pool, size] : pools) {
        INFO() << "Pool " << pool.id() << " : " << size << " : " << pool.privateName();
    }
}

bool Client::archivePool(PoolId poolId) const
{
    auto pool = loadPool(client_, poolId);
    INFO() << "Pool id: " << pool.id() << " " << pool.status() << " : " << pool.privateName();
    if (pool.status() != io::PoolStatus::Closed) {
        ERROR() << "Pool " << pool.id() << " can not be archived in status: " << pool.status();
        return false;
    }

    // without retries
    auto operation = client_.archivePool(pool.id());
    INFO() << "Pool " << pool.id() << " archivation id: " << operation.id();
    return true;
}

void Client::dumpPool(PoolId poolId) const
{
    auto pool = loadPool(client_, poolId);
    INFO() << "Pool id: " << pool.id() << " " << pool.status() << " : " << pool.privateName();
}

Assignments Client::loadAssignments(PoolId poolId) const
{
    auto response = loadAcceptedAssignments(client_, projectId_, std::to_string(poolId));
    Assignments result;
    for (const auto& assignment : response.assignments()) {
        result.emplace_back(makeAssignment(poolId, assignment));
    }
    return result;
}

Attachments Client::loadAttachments(const AttachmentIds& attachmentIds) const
{
    Attachments result;
    for (const auto& attachmentId : attachmentIds) {
        Attachment attachment;
        attachment.id = attachmentId;
        attachment.data = loadAttachment(client_, attachmentId);
        INFO() << "Loaded attachment id: " << attachmentId << " size: " << attachment.data.size();
        result.emplace_back(std::move(attachment));
    }
    return result;
}

} // namespace maps::wiki::yang
