#include "taskdata.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/stringutils/include/split.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/serialization.h>
#include <yandex/maps/wiki/common/string_utils.h>

namespace maps::wiki::validation {

namespace {

template <class T>
std::vector<T> parseArray(const std::string& str) {
    if (str.empty()) {
        return {};
    }
    REQUIRE(str.size() >= 2 && str.front() == '{' && str.back() == '}',
        "Unrecognized array value format: " << str);
    return stringutils::fields<T>(str.substr(1, str.size() - 2), ',');
}

} // namespace


TaskData::TaskData(pqxx::transaction_base& txnCore, DBID taskId)
    : taskId(taskId)
{
    auto query =
        "SELECT"
        "   *,"
        "   ST_AsBinary(vt.aoi_geom) AS the_geom,"
        "   COALESCE("
        "       (LENGTH(REPLACE(log, yt_operation_id, '~')) - LENGTH(REPLACE(log, yt_operation_id, ''))),"
        "       0) AS start_count"
        " FROM service.validation_task vt"
        " JOIN service.task t USING(id)"
        " WHERE id=" + std::to_string(taskId);

    auto result = txnCore.exec(query);
    ASSERT(result.size() == 1);

    const auto& row = result[0];

    grinderTaskId = row["grinder_task_id"].as<std::string>({});
    ytOperationId = row["yt_operation_id"].as<std::string>({});

    checks = parseArray<validator::TCheckId>(row["checks"].as<std::string>({}));
    aoiIds = parseArray<DBID>(row["aoi_ids"].as<std::string>({}));
    parentTaskId = row["parent_id"].as<DBID>(0);
    branchId = row["branch_id"].as<DBID>();
    commitId = row["commit_id"].as<DBID>();
    regionId = row["region_id"].as<DBID>(0);
    uid = row["created_by"].as<Uid>();
    aoiBuffer = row["aoi_buffer"].as<double>(0);
    isHeavy = row["is_heavy"].as<bool>(false);
    onlyChangedObjects = row["only_changed_objects"].as<bool>(false);
    startCount = row["start_count"].as<size_t>();

    std::string geomWkb = pqxx::binarystring(row["the_geom"]).str();
    if (!geomWkb.empty()) {
        std::stringstream geomWkbStream(geomWkb);
        aoiGeom = std::make_shared<geolib3::Polygon2>(
            geolib3::WKB::read<geolib3::Polygon2>(geomWkbStream));
    }
}

bool TaskData::hasCheck(validator::TCheckId check) const
{
    return std::find(checks.begin(), checks.end(), check) != checks.end();
}

bool TaskData::canSplit() const
{
    return
        isHeavy &&
        !aoiGeom &&
        aoiIds.empty() &&
        !regionId &&
        !parentTaskId &&
        !onlyChangedObjects;
}

void TaskData::dump() const
{
    INFO() << "Taskdata:"
        << " grinder: " << grinderTaskId
        << " yt: " << ytOperationId
        << " id: " << taskId
        << " parent: " << parentTaskId
        << " uid: " << uid
        << " branch: " << branchId
        << " commit: " << commitId
        << " region: " << regionId
        << " geom: " << (aoiGeom ? "polygon" : "null")
        << " buffer: " << aoiBuffer
        << " heavy: " << isHeavy
        << " only-changed-objects: " << onlyChangedObjects
        << " aoi: { " << wiki::common::join(aoiIds, ',') << " }"
        << " checks: { " << wiki::common::join(checks, ',') << " }"
        << " start-count: " << startCount;
}

} // namespace maps::wiki::validation
