#include "import_entrances.h"

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/serialization.h>

#include <boost/algorithm/string/split.hpp>

#include <exception>
#include <fstream>
#include <memory>
#include <set>
#include <string>
#include <vector>

namespace mwb = maps::wiki::business_entrance_import_tool;
namespace mwc = maps::wiki::common;
namespace mwr = maps::wiki::revision;

namespace {

std::set<mwr::DBID>
getAreaIds(const std::string& areaIdsStr)
{
    if (areaIdsStr.empty()) {
        return {};
    }
    std::vector<std::string> parts;
    auto delimiter = [](char c) { return c == ','; };
    boost::split(parts, areaIdsStr, delimiter, boost::algorithm::token_compress_on);

    std::set<mwr::DBID> result;
    for (const auto& part : parts) {
        result.insert(std::stoull(part));
    }
    return result;
}

std::string json(const mwb::Entrances& entrances)
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder object) {
        object["type"] = "FeatureCollection";
        object["crs"] << [&](maps::json::ObjectBuilder crs) {
            crs["type"] = "name";
            crs["properties"] << [&](maps::json::ObjectBuilder properties) {
                properties["name"]   = "urn:ogc:def:crs:OGC:1.3:CRS84";
            };
        };
        object["features"] << [&](maps::json::ArrayBuilder array) {
            for (const auto& entrance : entrances) {
                array << [&](maps::json::ObjectBuilder feature) {
                    feature["type"] = "Feature";
                    feature["properties"] << [&](maps::json::ObjectBuilder properties) {
                        properties["type"] = "business";

                        std::map<std::string, std::set<mwr::DBID>> catIdToFtIds;
                        for (const auto& poi : entrance.pois) {
                            catIdToFtIds[poi.categoryId].insert(poi.ftId);
                        }
                        for (const auto& kvp : catIdToFtIds) {
                            const auto catId = kvp.first;
                            const auto& ftIds = kvp.second;
                            properties[catId] << [&](maps::json::ArrayBuilder catPoiBuilder) {
                                for (const auto ftId : ftIds) {
                                    catPoiBuilder << ftId;
                                }
                            };
                        }
                    };
                    feature["geometry"] = maps::geolib3::geojson(
                        maps::geolib3::convertMercatorToGeodetic(entrance.pos));
                };
            }
        };
    };
    return builder.str();
}

} // namespace

int main(int argc, char** argv) try
{
    maps::cmdline::Parser argsParser;
    auto areaIdList = argsParser.string("area-ids")
        .required()
        .help("Comma separated area object ID list");
    auto configPath = argsParser.string("config")
        .defaultValue("")
        .help("Config path (default: path to services.xml)");
    auto resultDir = argsParser.string("result-dir")
        .required()
        .help("Result directory path");

    argsParser.parse(argc, argv);

    auto config = configPath.empty()
        ? mwc::loadDefaultConfig()
        : std::make_unique<mwc::ExtendedXmlDoc>(configPath);
    mwc::PoolHolder dbHolder(*config, "long-read", "long-read");
    auto& pgPool = dbHolder.pool();

    auto areaIds = getAreaIds(areaIdList);
    auto entrancesToImport = mwb::getEntrancesToImport(areaIds, pgPool);

    auto resultFile = resultDir + "/poi_entrance.json";
    std::ofstream fs(resultFile);
    REQUIRE(!fs.fail(), "Couldn't open '" << resultFile << "' for writing");
    fs << json(entrancesToImport);
    fs.close();

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << "Exception caught: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Exception caught: " << ex.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown exception caught";
    return EXIT_FAILURE;
}
