#include "yt_reader.h"

#include <yandex/maps/wiki/common/yt.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/log8/include/log8.h>

#include <mapreduce/yt/interface/client.h>
#include <sprav/protos/export.pb.h>

#include <set>
#include <string>
#include <vector>

using namespace NYT;

namespace maps::wiki::business_entrance_import_tool {

namespace {

const TString YT_REALM = "hahn";
const std::string YT_NYAK_MAPPING = "//home/altay/db/export/current-state/snapshot/nyak_mapping";

const std::set<uint64_t> MALL_RUBRICS = {184107509, 184108083};

} // namespace

YtData readOrganizationsDataFromYt()
{
    NYT::JoblessInitialize();
    auto client = common::yt::createYtClient(YT_REALM);
    auto rowCount = client->Get(TString(YT_NYAK_MAPPING + "/@row_count")).AsInt64();
    auto reader = client->CreateTableReader<NYT::TNode>(TString(YT_NYAK_MAPPING));

    INFO() << "Reading organizations from YT started";

    YtData result;
    result.orgs.reserve(rowCount);

    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        if (row["export_proto"].IsNull()) {
            continue;
        }
        NSpravExport::TExportedCompany company;
        company.ParseFromString(row["export_proto"].AsString());

        for (const auto& rubric : company.GetRubric()) {
            if (rubric.HasIsMain() && rubric.GetIsMain() &&
                MALL_RUBRICS.contains(rubric.GetId()))
            {
                result.mallPermalinks.insert(row["permalink"].IntCast<PermalinkId>());
                break;
            }
        }

        if (!company.HasGeo()) {
            continue;
        }

        auto orgFtId = std::stoull(row["original_id"].AsString());
        auto orgPos = geolib3::convertGeodeticToMercator(geolib3::Point2(
            company.GetGeo().GetLocation().GetPos().GetLon(),
            company.GetGeo().GetLocation().GetPos().GetLat()));

        std::vector<geolib3::Point2> orgEntrancePos;
        for (const auto& entrance : company.GetGeo().GetLocation().GetEntrance()) {
            auto entrancePos = geolib3::convertGeodeticToMercator(geolib3::Point2(
                entrance.GetPos().GetLon(),
                entrance.GetPos().GetLat()));
            orgEntrancePos.push_back(entrancePos);
        }

        std::vector<PermalinkId> orgLocatedAt;
        for (const auto& relation : company.GetRelation()) {
            if (relation.GetType() == NSpravExport::TRelation::LocatedAt)
            {
                orgLocatedAt.push_back(relation.GetPermalink());
            }
        }

        result.orgs.push_back({orgFtId, orgPos, orgEntrancePos, orgLocatedAt});
    }

    INFO() << "Reading organizations from YT finished";
    INFO() << "Total organizations count: " << result.orgs.size();
    INFO() << "Total malls count: " << result.mallPermalinks.size();

    return result;
}

} // maps::wiki::business_entrance_import_tool
