#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/configs/editor/config_holder.h>
#include <yandex/maps/wiki/configs/editor/category_groups.h>
#include <yandex/maps/wiki/social/gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <maps/wikimap/mapspro/tools/category_group_edits_stat/lib/structs.h>

#include <algorithm>
#include <cassert>
#include <iostream>

using namespace maps;
using namespace maps::wiki::common;
namespace json = maps::json;
namespace social = maps::wiki::social;
namespace cfgeditor = maps::wiki::configs::editor;

bool isEditCreated(const social::Event& event)
{
    return event.action() == "object-created";
}

std::optional<CatGroup>
commitCategoryGroup(
    const social::Event& event,
    const cfgeditor::CategoryGroups& categoryGroups)
{
    if (auto category = event.getPrimaryObjectCategory(); category) {
        auto group = categoryGroups.findGroupByCategoryId(category.value());
        return group ? std::make_optional(group->id()) : std::nullopt;
    } else if (event.action().starts_with("group")) {
        return "groupedits_group";
    } else {
        return std::nullopt;
    }
}

std::string toJsonString(const std::pair<UidGroup, EditsStat>& statRecord)
{
    json::Builder builder;
    builder << [&](json::ObjectBuilder builder) {
        builder["uid"] << statRecord.first.uid;
        builder["group"] << statRecord.first.group;
        builder["total"] << statRecord.second.total;
        builder["created"] << statRecord.second.created;
    };
    return builder.str();
}

void printStat(const std::map<UidGroup, EditsStat>& stat)
{
    for (const auto& [key, val] : stat) {
        std::cout << "uid: " << key.uid << ", group: " << key.group << std::endl;
        std::cout << "    total: " << val.total << ", crated: " << val.created << std::endl;
    }
}

void applyEventsToStat(
    const social::Events& events,
    const cfgeditor::CategoryGroups& categoryGroups,
    std::map<UidGroup, EditsStat>& stat)
{
    for (const auto& event : events) {
        auto group = commitCategoryGroup(event, categoryGroups);
        if (!group) {
            continue;
        }
        UidGroup key{event.createdBy(), group.value()};
        auto& val = stat[key];
        val.total++;
        if (isEditCreated(event)) {
            val.created++;
        }
    }
}

int main(int argc, char* argv[]) try
{
   // Parsing args
   //
    cmdline::Parser parser;

    auto servicesConfigArg = parser.string("config-services")
        .help("Path to service configuration file");

    auto configEditorPathArg = parser.string("config-editor")
        .required()
        .help("Path to config editor file");

    auto minCommitIdArg = parser.size_t("min-commit-id")
        .required()
        .help("Calculate stat for range starting from this commit id");

    auto maxCommitIdArg = parser.size_t("max-commit-id")
        .required()
        .help("Calculate stat for range ending at this commit id");

    auto batchSizeArg = parser.size_t("batch-size")
        .required()
        .help("Events load batch size");

    parser.parse(argc, argv);

    size_t minCommitId = minCommitIdArg;
    size_t maxCommitId = maxCommitIdArg;
    size_t batchSize = batchSizeArg;
    ASSERT(minCommitId <= maxCommitId);

    INFO() << "Starting script for commits range ["
           << minCommitId << ", " << maxCommitId << "]";

    const auto servicesConfig = [&]() {
        if (servicesConfigArg.defined()) {
            return std::make_unique<ExtendedXmlDoc>(servicesConfigArg);
        } else {
            return loadDefaultConfig();
        }
    }();

    cfgeditor::ConfigHolder editorConfig(configEditorPathArg);
    const auto& categoryGroups = editorConfig.categoryGroups();

    // Loading and processing commit events
    //
    PoolHolder socialTxnPool(*servicesConfig, "social", "grinder");
    auto socialTxnHandle = socialTxnPool.pool().slaveTransaction();
    social::Gateway gtw(socialTxnHandle.get());

    std::map<UidGroup, EditsStat> stat;

    size_t curCommitIdStart = minCommitId;
    while (curCommitIdStart <= maxCommitId) {
        long long curCommitIdEnd = std::min(curCommitIdStart + batchSize - 1, maxCommitId);

        INFO() << "Loading events for commit range ["
               << curCommitIdStart << ", " << curCommitIdEnd << "]";

        auto events = gtw.loadTrunkEditEventsByCommitRange(curCommitIdStart, curCommitIdEnd);
        INFO() << "Loaded events size: " << events.size();

        INFO() << "Applying events to stat";
        applyEventsToStat(events, categoryGroups, stat);

        curCommitIdStart = curCommitIdEnd + 1;
    }

    for (const auto& statRecord : stat) {
        std::cout << toJsonString(statRecord) << "\n";
    }

    return EXIT_SUCCESS;

} catch (const Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    ERROR() << "Unknown exeption";
    return EXIT_FAILURE;
}
