#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/social/edits_stat.h>
#include <yandex/maps/wiki/social/edits_stat_gateway.h>
#include <yandex/maps/wiki/social/gateway.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <maps/wikimap/mapspro/tools/category_group_edits_stat/lib/structs.h>

#include <algorithm>
#include <cassert>
#include <iostream>

using namespace maps;
using namespace maps::wiki::common;
namespace json = maps::json;
namespace social = maps::wiki::social;
namespace sql_chemistry = maps::sql_chemistry;

void lockCatGroupEditsTableForUpdates(pqxx::transaction_base& socialTxn)
{
    std::string query =
        "LOCK TABLE social.cat_group_edits_stat IN EXCLUSIVE MODE";
    socialTxn.exec(query);
}

std::pair<UidGroup, EditsStat> fromJson(const std::string& json)
{
    json::Value value = json::Value::fromString(json);
    ASSERT(value.isObject());

    return std::make_pair(
        UidGroup{value["uid"].as<TUid>(), value["group"].as<std::string>()},
        EditsStat{value["total"].as<size_t>(), value["created"].as<size_t>()}
    );
}

int main(int argc, char* argv[]) try
{
    // Parsing args
    //
    cmdline::Parser parser;

    auto servicesConfigArg = parser.string("config-services")
        .help("Path to service configuration file");

    parser.parse(argc, argv);

    const auto servicesConfig = [&]() {
        if (servicesConfigArg.defined()) {
            return std::make_unique<ExtendedXmlDoc>(servicesConfigArg);
        } else {
            return loadDefaultConfig();
        }
    }();

    // Loading and user stat records
    //
    std::map<UidGroup, EditsStat> stat;
    std::string curLine;
    while (std::cin >> curLine) {
        stat.insert(fromJson(curLine));
        curLine.clear();
    }

    PoolHolder socialTxnPool(*servicesConfig, "social", "grinder");
    auto socialTxnHandle = socialTxnPool.pool().masterWriteableTransaction();

    lockCatGroupEditsTableForUpdates(socialTxnHandle.get());

    social::CatGroupEditsStatGateway gtw(socialTxnHandle.get());

    int recordInd = 0;
    for (const auto& [uidGroup, editsStat] : stat) {
        // Update single record
        //
        auto recordExisting = gtw.tryLoadOne(
            social::table::CatGroupEditsStatTbl::uid == uidGroup.uid &&
            social::table::CatGroupEditsStatTbl::categoryGroup == uidGroup.group
        );

        social::CatGroupEditsStat recordToInsert{
            0,
            uidGroup.uid,
            uidGroup.group,
            editsStat.total,
            editsStat.created
        };
        if (recordExisting) {
            recordToInsert.id = recordExisting->id;
            recordToInsert.totalCount   += recordExisting->totalCount;
            recordToInsert.createdCount += recordExisting->createdCount;
        }

        gtw.upsert(recordToInsert);

        // Log progress
        //
        recordInd++;
        if (recordInd % 100) {
            INFO() << "Writed " << recordInd << " records";
        }
    }

    INFO() << "Committing records";
    socialTxnHandle->commit();

    return EXIT_SUCCESS;

} catch (const Exception& e) {
    ERROR() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << e.what();
    return EXIT_FAILURE;
} catch (...) {
    ERROR() << "Unknown exeption";
    return EXIT_FAILURE;
}
