#include "yt_reader.h"

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>

#include <exception>
#include <fstream>
#include <iostream>
#include <string>

namespace mww = maps::wiki::easyview_walkers_photos_tool;

namespace {

constexpr size_t DOUBLE_PRECISION = 12;
const auto RECENT_PHOTO_AGE = std::chrono::days(14);

void makeEasyViewFile(
    const mww::PhotosData& photosData,
    const std::string& resultFile)
{
    std::map<std::string, mww::PhotosData> photosDataByPermalink;

    const auto now = maps::chrono::TimePoint::clock::now();

    for (size_t i = 0; i < photosData.size(); ++i) {
        const auto& photoData = photosData[i];
        if (!photoData.coordObject || !photoData.coordTaking) {
            continue;
        }
        photosDataByPermalink[photoData.permalink].push_back(photoData);
    }

    std::ofstream fs(resultFile);
    REQUIRE(!fs.fail(), "Couldn't open '" << resultFile << "' for writing");

    fs.precision(DOUBLE_PRECISION);

    for (const auto& [permalink, photosData] : photosDataByPermalink) {
        if (!permalink.empty()) {
            for (size_t i = 0; i < photosData.size() - 1; ++i) {
                fs << "!linestyle=blue:2" << std::endl;
                fs << photosData[i].coordObject->x() << ";" << photosData[i].coordObject->y() << ";"
                   << photosData[i + 1].coordObject->x() << ";" << photosData[i + 1].coordObject->y() << std::endl;
            }
        }

        size_t size = photosData.size() > 1 ? 4 : 2;
        for (const auto& photoData : photosData) {
            fs << (now - photoData.submitTs <= RECENT_PHOTO_AGE
                   ? "!pointstyle=red:red:"
                   : "!pointstyle=green:green:")
               << size << std::endl;
            fs << photoData.coordObject->x() << ";" << photoData.coordObject->y() << ";"
               << photoData.tag << "<br/>"
               << "<a href=\"" << photoData.urlPhoto << "\">" << photoData.urlPhoto << "</a><br/>"
               << "<a href=\"" << photoData.urlPermalink << "\">" << photoData.urlPermalink << "</a>" << std::endl;
        }

        for (const auto& photoData : photosData) {
            fs << (now - photoData.submitTs <= RECENT_PHOTO_AGE
                   ? "!linestyle=red:2"
                   : "!linestyle=green:2")
               << std::endl;
            fs << photoData.coordTaking->x() << ";" << photoData.coordTaking->y() << ";"
               << photoData.coordObject->x() << ";" << photoData.coordObject->y() << ";"
               << photoData.tag << "<br/>"
               << "<a href=\"" << photoData.urlPhoto << "\">" << photoData.urlPhoto << "</a><br/>"
               << "<a href=\"" << photoData.urlPermalink << "\">" << photoData.urlPermalink << "</a>" << std::endl;
        }
    }

    fs.close();
}

} // namespace

int main(int argc, char** argv) try
{
    maps::cmdline::Parser argsParser;
    auto resultFile = argsParser.string("result-file")
        .required()
        .help("Result file path");
    auto limit = argsParser.num("limit")
        .help("Max photos to obtain");
    auto ignoreOldPhotos = argsParser.flag("ignore-old-photos")
        .help("Ignore photos more than 1 year old");

    argsParser.parse(argc, argv);

    auto photosData = mww::readPhotoDataFromYt(
        limit, ignoreOldPhotos);
    makeEasyViewFile(photosData, resultFile);

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "Exception caught: " << e << std::endl;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    std::cout << "Exception caught: " << ex.what() << std::endl;
    return EXIT_FAILURE;
} catch (...) {
    std::cout << "Unknown exception caught" << std::endl;
    return EXIT_FAILURE;
}
