#include "worker.h"
#include "fbapi_task_wrapper.h"
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/libs/geolib/include/point.h>

#include <yandex/maps/geolib3/proto.h>

// https://a.yandex-team.ru/arc/trunk/arcadia/maps/doc/proto/yandex/maps/proto/nmaps/feedback/api.proto
#include <yandex/maps/proto/nmaps/feedback/api.pb.h>

#include <fstream>
#include <iostream>

using namespace std::string_literals;

namespace api = yandex::maps::proto::nmaps::feedback::api;

namespace maps::wiki::fbapi_social_tool {

namespace {

const auto SYNC_TOKEN = "X-Ya-Sync-Token"s;

std::string makeBaseUrl(const std::string& url)
{
    return url + (*url.rbegin() == '/' ? "" : "/");
}

} // namespace

Worker::Worker(
        const std::string& socialBackofficeUrl,
        const std::string& avatarsUrl,
        std::string externalIdPrefix,
        const std::string& fbapiConnStr)
    : socialBackofficeUrl_(makeBaseUrl(socialBackofficeUrl))
    , avatarsUrl_(makeBaseUrl(avatarsUrl))
    , externalIdPrefix_(std::move(externalIdPrefix))
    , fbapiConn_(fbapiConnStr)
{
    ASSERT(!socialBackofficeUrl_.empty());
    ASSERT(!avatarsUrl_.empty());
    ASSERT(!externalIdPrefix_.empty());
}

void Worker::run(const std::string& idsFile) const
{
    auto reader = [&](std::istream& infile)
    {
        std::string str;
        while (std::getline(infile, str)) {
            if (!str.empty()) {
                process(str);
            }
        }
    };

    if (idsFile == "-") {
        reader(std::cin);
    } else {
        std::ifstream infile(idsFile);
        REQUIRE(infile, "input ids file not exists: " << idsFile);
        reader(infile);
    }
}

void Worker::process(const std::string& id) const
{
    INFO() << "Processing fbapi id: " << id;
    pqxx::read_transaction txn(fbapiConn_);

    auto query =
        "SELECT original_task"
        " FROM public.feedback_task"
        " WHERE service = 'sprav' AND id = " + txn.quote(id);
    try {
        auto result = txn.exec(query);
        REQUIRE(!result.empty(), "Task not found");

        auto jsonStr = result[0][0].as<std::string>();
        submit(id, makeHypothesis(json::Value::fromString(jsonStr)));
    } catch (maps::Exception& ex) {
        ERROR() << "Error on proccess task, id: " << id << " : " << ex;
    } catch (std::exception& ex) {
        ERROR() << "Error on proccess task, id: " << id << " : " << ex.what();
    }
}

std::string Worker::makeHypothesis(const json::Value& originalTask) const
{
    FbapiTaskWrapper task(originalTask);

    const auto companyId = task.companyId();
    const auto position = task.position();

    api::Hypothesis hypothesis;
    auto* changeBusiness = hypothesis.mutable_change_business();
    INFO() << "Company id (permalink): " << companyId;
    changeBusiness->set_company_id(TString(companyId));

    auto* company = changeBusiness->mutable_company();
    INFO() << "Position: " << std::to_string(position.x()) << " " << std::to_string(position.y());
    *(company->mutable_point()) = geolib3::proto::encode(position);

    for (const auto& entPosition : task.entrances()) {
        INFO() << "Entrance position: " << std::to_string(entPosition.x()) << " " << std::to_string(entPosition.y());
        auto* entrance = company->add_entrances();
        *(entrance->mutable_point()) = geolib3::proto::encode(entPosition);
    }

    auto comment = task.comment();
    if (!comment.empty()) {
        INFO() << "Comment: " << comment;
        auto* context = changeBusiness->mutable_context();
        context->set_comment(TString(comment));
    }

    for (const auto& photo : task.attachedPhotos()) {
        auto url = makeAvatarsUrl(photo);
        INFO() << "Photo: " << url;
        auto* context = changeBusiness->mutable_context();
        context->add_files()->set_url(TString(url));
    }

    INFO() << "To social-backoffice: " << hypothesis.DebugString();
    return hypothesis.SerializeAsString();
}

std::string Worker::makeAvatarsUrl(const std::string& key) const
{
    return avatarsUrl_ + "get-maps-feedback/" + key + "/orig";
}

void Worker::submit(const std::string& id, const std::string& body) const
{
    const auto externalReferenceId = externalIdPrefix_ + ":" + id;
    std::string token;

    auto makeHttpRequest = [&] (const auto& uri, const auto& method) {
        http::URL url(socialBackofficeUrl_ + uri);
        url.addParam("id", externalReferenceId);
        INFO() << "Url: " << url;

        http::Request request(httpClient_, method, url);
        if (!token.empty()) {
            request.addHeader(SYNC_TOKEN, token);
        }
        return request;
    };

    {
        auto request = makeHttpRequest("v1/hypothesis/add", http::POST);
        request.setContent(body);
        auto response = request.perform();
        const auto status = response.status();
        INFO() << "Add: " << externalReferenceId << " : http status: " << status;
        if (status == 201) {
            token = response.header(SYNC_TOKEN);
        } else {
            ERROR() << "Body: " << response.readBody();
        }
    }
    {
        auto request = makeHttpRequest("v1/hypothesis/get", http::GET);
        auto response = request.perform();
        const auto status = response.status();
        INFO() << "Get: " << externalReferenceId << " : http status: " << status;
        if (status == 200) {
            api::Hypothesis hypothesis;
            ASSERT(hypothesis.ParseFromString(TString(response.readBody())));
            INFO() << "From social-backoffice: " << hypothesis.DebugString();
        }
    }
    {
        auto request = makeHttpRequest("v1/hypothesis/result", http::GET);
        auto response = request.perform();
        const auto status = response.status();
        INFO() << "Result: " << externalReferenceId << " : http status: " << status;
        if (status == 200) {
            api::HypothesisResult hypothesisResult;
            ASSERT(hypothesisResult.ParseFromString(TString(response.readBody())));
            INFO() << "Result from social-backoffice: " << hypothesisResult.DebugString();
        }
    }
}

} // namespace maps::wiki::fbapi_social_tool
