#include "route.h"
#include "serialize.h"

#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>

#include <iostream>
#include <fstream>

namespace po = boost::program_options;


void
clusterize(
    maps::wiki::ft_fix::RoutesData& routesData,
    const std::string& fileName,
    double thresholdMeters,
    uint64_t idStartFrom)
{
    auto routes = routesData.clusterize(thresholdMeters);
    std::ofstream os(fileName);
    serializeJson(os, routes, idStartFrom);
}

static const std::string OPT_HELP = "help";
static const std::string OPT_INPUT_FILE = "input";
static const std::string OPT_OUTPUT_FILE = "output";
static const std::string OPT_THRESHOLD = "threshold";
static const std::string OPT_ID_START_FROM = "id-start-from";


int
main(int argc, char* argv[])
{
    std::string stopsFileName;
    std::string jsonFileName;
    double thresholdMeters;
    uint64_t idStartFrom;

    po::options_description desc("Allowed options");
    desc.add_options()
        (OPT_HELP.c_str(),
            "produce help message")
        (OPT_INPUT_FILE.c_str(), po::value<std::string>(&stopsFileName),//->required(),
            "input file")
        (OPT_OUTPUT_FILE.c_str(), po::value<std::string>(&jsonFileName),
            "output json file")
        (OPT_THRESHOLD.c_str(), po::value<double>(&thresholdMeters)->default_value(10000.0),
            "route threshold, meters")
        (OPT_ID_START_FROM.c_str(), po::value<size_t>(&idStartFrom),
            "initial json id for new objects");

    boost::program_options::variables_map vm;
    try {
        po::store(po::parse_command_line(argc, argv, desc), vm);
    } catch (po::error& e) {
        std::cerr << e.what() << std::endl;
        return 1;
    }
    po::notify(vm);

    if (argc == 1 || vm.count(OPT_HELP)
        || vm.count(OPT_INPUT_FILE) == 0
        || vm.count(OPT_OUTPUT_FILE) == 0
        || vm.count(OPT_ID_START_FROM) == 0)
    {
        std::cerr << "Usage: routefixer <options>" << std::endl;
        std::cerr << desc << std::endl;
        return 1;
    }


    std::ifstream ifs(stopsFileName);
    maps::wiki::ft_fix::RoutesData routesData(ifs);

    clusterize(routesData, jsonFileName, thresholdMeters, idStartFrom);

    return 0;
}

