#include "piste_ids.h"

#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <exception>
#include <iostream>

namespace mwc = maps::wiki::common;
namespace rdr = maps::garden::modules::renderer_denormalization;

namespace {

std::unordered_set<size_t>
filterPisteIdsByCategory(
    maps::pgpool3::TransactionHandle& coreTxn,
    const std::unordered_set<size_t>& ids)
{
    const auto query = "SELECT DISTINCT object_id "
        "FROM revision.object_revision "
        "JOIN revision.attributes ON (attributes_id = id) "
        "WHERE object_id IN (" + mwc::join(ids, ',') + ") "
        "  AND contents ? 'cat:rd'";
    const auto rows = coreTxn->exec(query);

    std::unordered_set<size_t> result;
    for (const auto& row : rows) {
        auto pisteId = row["object_id"].as<size_t>();
        std::cout << pisteId << std::endl;
        result.insert(pisteId);
    }
    return result;
}

size_t
checkAndInsertRevAttr(
    maps::pgpool3::TransactionHandle& coreTxn,
    const std::string& tableName,
    const std::vector<std::string>& hstoreArray,
    const size_t id = 0)
{
    ASSERT(hstoreArray.size() % 2 == 0);
    const auto hstoreValue =
        "hstore(ARRAY[" +
        mwc::join(
            hstoreArray,
            [&](const std::string& str){ return coreTxn->quote(str); },
            ',') +
        "])";

    const auto rowsSelect = coreTxn->exec(
        "SELECT id FROM " + tableName +
        " WHERE contents = " + hstoreValue +
        (id ? " AND id = " + std::to_string(id) : ""));

    ASSERT(rowsSelect.size() <= 1);
    if (rowsSelect.size() == 1) {
        return rowsSelect[0][0].as<size_t>();
    }

    const auto rowsInsert = coreTxn->exec(
        "INSERT INTO " + tableName +
        " (" + (id ? "id, " : "") + "contents)"
        " VALUES (" +
        (id ? std::to_string(id) + ", " : "") +
        hstoreValue + ") RETURNING id");

    ASSERT(rowsInsert.size() == 1);
    return rowsInsert[0][0].as<size_t>();
}

void convertSportTracks(maps::pgpool3::Pool& pgPool)
{
    auto coreTxn = pgPool.masterWriteableTransaction();

    /* Add ft_type revision attributes */

    auto insertRevAttrFtType = [&](const int ftTypeId) {
        return checkAndInsertRevAttr(
            coreTxn,
            "revision.attributes",
            {"cat:sport_track", "1", "sport_track:ft_type_id", std::to_string(ftTypeId)});
    };

    const auto revAttrGreen = insertRevAttrFtType(2901);
    const auto revAttrBlue = insertRevAttrFtType(2902);
    const auto revAttrRed = insertRevAttrFtType(2903);
    const auto revAttrBlack = insertRevAttrFtType(2904);
    const auto revAttrDottedBlack = insertRevAttrFtType(2905);
    // const auto revAttrSki = insertRevAttrFtType(2906); // not presented in piste_ids.h
    const auto revAttrTubing = insertRevAttrFtType(2907);
    const auto revAttrSnowpark = insertRevAttrFtType(2908);

    /* Add relations revision attributes */

    auto insertRevAttrRelation = [&](const std::vector<std::string>& hstoreArray)
    {
        auto result = checkAndInsertRevAttr(coreTxn, "revision.attributes", hstoreArray);
        checkAndInsertRevAttr(coreTxn, "revision.attributes_relations", hstoreArray, result);
        return result;
    };

    /* Collect all piste ids */

    const auto greenPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_GREEN_IDS);
    const auto bluePisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_BLUE_IDS);
    const auto redPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_RED_IDS);
    const auto blackPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_BLACK_IDS);
    const auto dottedBlackPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_DOTTED_BLACK_IDS);
    const auto tubingPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_TUBING_IDS);
    const auto snowparkPisteIds = filterPisteIdsByCategory(coreTxn, rdr::PISTE_SNOWPARK_IDS);

    std::unordered_set<size_t> allPisteIds;
    allPisteIds.insert(greenPisteIds.begin(), greenPisteIds.end());
    allPisteIds.insert(bluePisteIds.begin(), bluePisteIds.end());
    allPisteIds.insert(redPisteIds.begin(), redPisteIds.end());
    allPisteIds.insert(blackPisteIds.begin(), blackPisteIds.end());
    allPisteIds.insert(dottedBlackPisteIds.begin(), dottedBlackPisteIds.end());
    allPisteIds.insert(tubingPisteIds.begin(), tubingPisteIds.end());
    allPisteIds.insert(snowparkPisteIds.begin(), snowparkPisteIds.end());

    /* Convert categories */

    auto updateCategory = [&](
        const size_t attributesId,
        const std::unordered_set<size_t>& ids)
    {
        coreTxn->exec(
            "UPDATE revision.object_revision_without_geometry "
            "SET attributes_id = " + std::to_string(attributesId) + " "
            "WHERE object_id IN (" + mwc::join(ids, ',') + ")");
    };

    updateCategory(revAttrGreen, greenPisteIds);
    updateCategory(revAttrBlue, bluePisteIds);
    updateCategory(revAttrRed, redPisteIds);
    updateCategory(revAttrBlack, blackPisteIds);
    updateCategory(revAttrDottedBlack, dottedBlackPisteIds);
    updateCategory(revAttrTubing, tubingPisteIds);
    updateCategory(revAttrSnowpark, snowparkPisteIds);

    /* Convert names */

    coreTxn->exec(
        "WITH names_ids AS ( "
        "  SELECT slave_object_id "
        "  FROM revision.object_revision_relation "
        "  JOIN revision.attributes ON (attributes_id = id) "
        "  WHERE master_object_id IN (" + mwc::join(allPisteIds, ',') + ") "
        "    AND contents -> 'rel:slave' = 'rd_nm' "
        ") "
        "UPDATE revision.attributes "
        "SET contents = hstore(ARRAY['cat:sport_track_nm', '1', 'sport_track_nm:lang', "
            "contents -> 'rd_nm:lang', 'sport_track_nm:name', contents -> 'rd_nm:name', 'sport_track_nm:is_local', contents -> 'rd_nm:is_local']) "
        "FROM revision.object_revision "
        "WHERE object_id IN (SELECT slave_object_id FROM names_ids) "
        "  AND attributes_id = id ");

    /* Convert relations where sport track is master */

    // Names
    auto revAttrNameOfficial = insertRevAttrRelation(
        {"rel:master", "sport_track", "rel:slave", "sport_track_nm", "rel:role", "official"});
    auto revAttrNameRenderLabel = insertRevAttrRelation(
        {"rel:master", "sport_track", "rel:slave", "sport_track_nm", "rel:role", "render_label"});
    auto revAttrNameSynonym = insertRevAttrRelation(
        {"rel:master", "sport_track", "rel:slave", "sport_track_nm", "rel:role", "synonym"});

    // sport_track --> rd_el
    auto revAttrRelationPart = insertRevAttrRelation(
        {"rel:master", "sport_track", "rel:slave", "rd_el", "rel:role", "sport_track_part"});

    auto updateSlaveRelations = [&](
        const size_t newAttributesId,
        const std::string& oldRole)
    {
        return coreTxn->exec(
            "UPDATE revision.object_revision_relation "
            "SET attributes_id = " + std::to_string(newAttributesId)  + " "
            "FROM revision.attributes "
            "WHERE master_object_id IN (" + mwc::join(allPisteIds, ',') + ") "
            "  AND attributes_id = id "
            "  AND contents -> 'rel:role' = " + coreTxn->quote(oldRole));
    };

    updateSlaveRelations(revAttrNameOfficial, "official");
    updateSlaveRelations(revAttrNameRenderLabel, "render_label");
    updateSlaveRelations(revAttrNameSynonym, "synonym");
    updateSlaveRelations(revAttrRelationPart, "part");

    /* Convert relations where sport track is slave */

    // ad --> sport_track
    auto revAttrRelationAssociatedWith = insertRevAttrRelation(
        {"rel:master", "ad", "rel:slave", "sport_track", "rel:role", "sport_track_associated_with"});

    auto updateMasterRelations = [&](
        const size_t newAttributesId,
        const std::string& oldRole)
    {
        return coreTxn->exec(
            "UPDATE revision.object_revision_relation "
            "SET attributes_id = " + std::to_string(newAttributesId)  + " "
            "FROM revision.attributes "
            "WHERE slave_object_id IN (" + mwc::join(allPisteIds, ',') + ") "
            "  AND attributes_id = id "
            "  AND contents -> 'rel:role' = " + coreTxn->quote(oldRole));
    };

    updateMasterRelations(revAttrRelationAssociatedWith, "associated_with");

    coreTxn->commit();
}

} // namespace

int main(int argc, char** argv) try
{
    maps::cmdline::Parser argsParser;
    auto configPath = argsParser.string("config")
        .required()
        .help("Config path (default: path to services.xml)");

    argsParser.parse(argc, argv);

    auto config = std::make_unique<mwc::ExtendedXmlDoc>(configPath);
    mwc::PoolHolder dbHolder(*config, "core", "editor");
    auto& pgPool = dbHolder.pool();

    convertSportTracks(pgPool);

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    std::cout << "Exception caught: " << e << std::endl;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    std::cout << "Exception caught: " << ex.what() << std::endl;
    return EXIT_FAILURE;
} catch (...) {
    std::cout << "Unknown exception caught" << std::endl;
    return EXIT_FAILURE;
}
