#include "client.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/retry_duration.h>
#include <util/string/cast.h>
#include <set>

using namespace std::string_literals;
using namespace maps::mrc::toloka;

namespace maps::wiki::toloka {

namespace {

const auto API_HOST = "toloka.yandex.ru"s;

const std::set<std::string> PHOTO_ARRAY_KEYS {
    "address_photo",
    "building_photo",
    "sign_photo",
};

std::string toColumnName(std::string key)
{
    std::transform(
        key.begin(),
        key.end(),
        key.begin(),
        [](unsigned char c) -> unsigned char {
            return c == ':' || c == '-' ? '_' : std::tolower(c);
        }
    );
    return key;
}

std::string timePointToStr(maps::chrono::TimePoint timePoint)
{
    auto str = maps::chrono::formatIsoDateTime(timePoint);
    return str.substr(0, str.find('Z'));
}

wiki::toloka::Assignment makeAssignment(PoolId poolId, const io::Assignment& assignment)
{
    wiki::toloka::Assignment a;

    a.add("assignment_link",
            "https://" + API_HOST + "/task/" + std::to_string(poolId) + "/" + assignment.id());
    a.add("assignment_assignment_id", assignment.id());
    a.add("assignment_worker_id", assignment.userId());
    a.add("assignment_status", (std::ostringstream() << assignment.status()).str());
    a.add("assignment_task_suite_id", assignment.taskSuiteId());
    a.add("assignment_started", timePointToStr(assignment.createdAt()));

    auto addOptionalTime = [&] (const auto& key, const auto& at) {
        if (at) {
            a.add(key, timePointToStr(*at));
        }
    };
    addOptionalTime("assignment_accepted", assignment.acceptedAt());
    addOptionalTime("assignment_submitted", assignment.submittedAt());

    const auto& tasks = assignment.tasks();
    ASSERT(tasks.size() == 1);
    const auto& task = tasks.front();
    a.add("assignment_task_id", task.id());
    const auto& inputValues = task.inputValues();
    ASSERT(inputValues.isObject());

    for (const auto& field : inputValues.fields()) {
        const auto& value = inputValues[field];
        a.add(
            toColumnName("input_" + field),
            value.isString()
                ? value.toString()
                : (std::ostringstream() << value).str());
    }

    const auto& solutions = assignment.solutions();
    ASSERT(solutions.size() == 1);
    const auto& outputValues = solutions.front().outputValues();
    ASSERT(outputValues.isObject());

    for (const auto& field : outputValues.fields()) {
        const auto& value = outputValues[field];
        if (value.isNull()) {
            WARN() << "Assignment: " << assignment.id()
                   << " Skip NULL value for field: " << field;
            continue;
        }
        auto column = toColumnName("output_" + field);
        if (value.isString()) {
            a.add(column, value.toString());
        } else {
            REQUIRE(PHOTO_ARRAY_KEYS.contains(field), "Unexpected non-string key: " << field);
            ASSERT(value.isArray());
            std::string str;
            for (const auto& v : value) {
                ASSERT(v.isString());
                str += (str.empty() ? "" : ",") + v.toString();
            }
            a.add(column, std::move(str));
        }
    }

    return a;
}

auto loadAssignmentsByStatus(
    const io::TolokaClient& client,
    ProjectId projectId,
    const std::string& poolId,
    auto status)
{
    io::Filter filter;
    filter.byProjectId(std::to_string(projectId));
    filter.byPoolId(poolId);
    filter.byAssignmentStatus(status);

    return common::retryDuration([&] {
        return client.getAssignments(filter);
    });
}

auto loadAcceptedAssignments(
    const io::TolokaClient& client,
    ProjectId projectId,
    const std::string& poolId)
{
    return loadAssignmentsByStatus(
        client, projectId, poolId, io::AssignmentStatus::Accepted);
}

auto loadSubmittedAssignments(
    const io::TolokaClient& client,
    ProjectId projectId,
    const std::string& poolId)
{
    return loadAssignmentsByStatus(
        client, projectId, poolId, io::AssignmentStatus::Submitted);
}

auto loadClosedPools(
    const io::TolokaClient& client,
    ProjectId projectId)
{
    io::Filter filter;
    filter.byProjectId(std::to_string(projectId));
    filter.byPoolStatus(io::PoolStatus::Closed);

    return common::retryDuration([&] {
        return client.getPools(filter);
    });
}

auto loadPool(
    const io::TolokaClient& client,
    PoolId poolId)
{
    return common::retryDuration([&] {
        return client.getPool(std::to_string(poolId));
    });
}

auto loadAttachment(
    const io::TolokaClient& client,
    const AttachmentId& attachmentId)
{
    return common::retryDuration([&] {
        return client.getAttachment(attachmentId);
    });
}

} // namespace

Client::Client(const std::string& oauthToken, ProjectId projectId)
    : client_(API_HOST, "OAuth " + oauthToken)
    , projectId_(projectId)
{
    REQUIRE(!oauthToken.empty(), "Empty OAuth TOKEN");
}

std::deque<PoolId> Client::getClosedPoolIds() const
{
    auto response = loadClosedPools(client_, projectId_);

    std::deque<PoolId> result;
    for (const auto& pool : response.pools()) {
        PoolId poolId = 0;
        REQUIRE(TryFromString(pool.id(), poolId),
                "Non-numeric pool id : " << pool.id());
        result.push_back(poolId);
    }
    return result;
}

void Client::listClosedPools() const
{
    auto response = loadClosedPools(client_, projectId_);

    struct PoolData {
        PoolData(io::Pool pool, size_t accepted, size_t submitted)
            : pool(std::move(pool))
            , accepted(accepted)
            , submitted(submitted)
        {}

        bool operator < (const PoolData& p) const
        {
            return pool.privateName() < p.pool.privateName();
        }

        io::Pool pool;
        size_t accepted;
        size_t submitted;
    };

    std::vector<PoolData> pools;
    for (const auto& pool : response.pools()) {
        auto responseAccepted = loadAcceptedAssignments(client_, projectId_, pool.id());
        auto responseSubmitted = loadSubmittedAssignments(client_, projectId_, pool.id());
        INFO() << "Analyzing pool " << pool.id() << " : " << pool.privateName();
        pools.emplace_back(
            pool,
            responseAccepted.assignments().size(),
            responseSubmitted.assignments().size());
    }

    std::sort(pools.begin(), pools.end());
    for (const auto& [pool, accepted, submitted] : pools) {
        INFO() << "Pool " << pool.id()
               << " : accepted: " << accepted
               << " : submitted: " << submitted
               << " : " << pool.privateName();
    }
}

bool Client::archivePool(PoolId poolId) const
{
    auto pool = loadPool(client_, poolId);
    INFO() << "Pool id: " << pool.id() << " " << pool.status() << " : " << pool.privateName();
    if (pool.status() != io::PoolStatus::Closed) {
        ERROR() << "Pool " << pool.id() << " can not be archived in status: " << pool.status();
        return false;
    }

    // without retries
    auto operation = client_.archivePool(pool.id());
    INFO() << "Pool " << pool.id() << " archivation id: " << operation.id();
    return true;
}

void Client::dumpPool(PoolId poolId) const
{
    auto pool = loadPool(client_, poolId);
    INFO() << "Pool id: " << pool.id() << " " << pool.status() << " : " << pool.privateName();
}

Assignments Client::loadAssignments(PoolId poolId) const
{
    Assignments result;

    auto appendData = [&] (const auto& response) {
        for (const auto& assignment : response.assignments()) {
            result.emplace_back(makeAssignment(poolId, assignment));
        }
    };

    const auto poolIdStr = std::to_string(poolId);
    appendData(loadAcceptedAssignments(client_, projectId_, poolIdStr));
    appendData(loadSubmittedAssignments(client_, projectId_, poolIdStr));
    return result;
}

Attachments Client::loadAttachments(const AttachmentIds& attachmentIds) const
{
    Attachments result;
    for (const auto& attachmentId : attachmentIds) {
        Attachment attachment;
        attachment.id = attachmentId;
        attachment.data = loadAttachment(client_, attachmentId);
        INFO() << "Loaded attachment id: " << attachmentId << " size: " << attachment.data.size();
        result.emplace_back(std::move(attachment));
    }
    return result;
}

} // namespace maps::wiki::toloka
