#!/usr/bin/env python

import argparse
import os
import time
from datetime import datetime, timedelta
from dateutil.parser import parse
from boto.s3.connection import S3Connection


def bucket_connection(bucket_name):
    ACCESS_KEY_ID = os.environ["AWS_ACCESS_KEY_ID"]
    SECRET_ACCESS_KEY = os.environ["AWS_SECRET_ACCESS_KEY"]
    ENDPOINT = os.environ["AWS_ENDPOINT"]

    conn = S3Connection(
        host=ENDPOINT,
        aws_access_key_id=ACCESS_KEY_ID,
        aws_secret_access_key=SECRET_ACCESS_KEY,
        )

    # Create new service's bucket
    return conn.get_bucket(bucket_name)


def to_gmt_datetime(dt):
    # 2018-11-26T10:34:26.000Z -> 2018-11-26 13:34:26
    [rt, _] = str(dt).replace('T', ' ').split('.')
    return parse(rt, ignoretz=True)


def to_local_datetime(dt):
    st = time.strptime(str(to_gmt_datetime(dt)), '%Y-%m-%d %H:%M:%S')
    lt = time.localtime(time.mktime(st) - time.timezone)
    return time.strftime('%Y-%m-%d %H:%M:%S', lt)


class S3Proxy:
    def __init__(self, args):
        self.args = args
        self._s3_conn = None

    def process_action(self):
        if self.args.action == 'get':
            self._get()
        elif self.args.action == 'put':
            self._put()
        elif self.args.action == 'copy':
            self._copy()
        elif self.args.action == 'info':
            self._info()
        elif self.args.action == 'del':
            self._del()
        elif self.args.action == 'list':
            self._list()
        elif self.args.action == 'clean':
            self._clean()
        else:
            raise Exception('Unknown action: ' + self.args.action)

    def _get_filepath(self):
        return self.args.filepath if self.args.filepath else os.path.basename(self.args.prefix)

    def _conn(self):
        if self._s3_conn is None:
            self._s3_conn = bucket_connection(self.args.bucket)
        return self._s3_conn

    def _retry_operation(self, operation, filepath):
        for i in range(1, self.args.attempts + 1):
            try:
                operation(self, filepath)
                return
            except Exception, ex:
                if i == self.args.attempts:
                    print filepath, "raise error"
                    raise
                print filepath, "suppress error: ", ex

    def _s3_url(self):
        return "s3://" + self.args.bucket + "/" + self.args.prefix

    def _get(self):
        def operation(self, filepath):
            if self.args.verbose:
                print "Get", filepath, "from", self._s3_url()
            key = self._conn().get_key(self.args.prefix)
            with open(filepath, "w") as f:
                key.get_file(f)

        self._retry_operation(operation, self._get_filepath())

    def _put(self):
        def operation(self, filepath):
            if self.args.verbose:
                print "Put", filepath, "into", self._s3_url()
            key = self._conn().new_key(self.args.prefix)
            with open(filepath, "r") as f:
                key.send_file(f, chunked_transfer=True)

        self._retry_operation(operation, self._get_filepath())

    def _copy(self):
        if not self.args.to:
            raise Exception('Unspecified destination prefix')
        self._get()
        self.args.prefix = self.args.to
        self._put()
        os.remove(self._get_filepath())

    def _info(self):
        key = self._conn().get_key(self.args.prefix)
        for k, v in key.__dict__.iteritems():
            if v is not None:
                print k + ":", v

    def _del(self):
        if self.args.dry_run:
            print "Dry run mode enabled"
        else:
            self._conn().delete_key(self.args.prefix)

    def _list(self):
        objects_count = 0
        objects_size = 0
        for o in self._conn().list(prefix=self.args.prefix):
            print to_local_datetime(o.last_modified), o.size, o.name
            objects_size += o.size
            objects_count += 1
            if self.args.limit > 0 and objects_count >= self.args.limit:
                break
        if self.args.verbose:
            def stat_size(size):
                result = str(size)
                for i in ["K", "M", "G", "T"]:
                    size = int(size / 1024)
                    result += " " + str(size) + i
                return result

            print "count:", objects_count, "size:", stat_size(objects_size)

    def _clean(self):
        expire_time = datetime.utcnow() - timedelta(days=self.args.keep)
        to_del = []
        verbose = self.args.limit == 0 or self.args.verbose
        analyze_limit = self.args.limit * 2 if self.args.incremental_clean else 0

        objects_count = 0
        for o in self._conn().list(prefix=self.args.prefix):
            if self.args.keep == 0 or to_gmt_datetime(o.last_modified) <= expire_time:
                if verbose:
                    print "to delete:", to_local_datetime(o.last_modified), o.size, o.name
                if self.args.limit > 0:
                    to_del.append(o.name)
                    if len(to_del) >= self.args.limit:
                        break
            objects_count += 1
            if analyze_limit > 0 and objects_count >= analyze_limit:
                break

        if len(to_del) > 0:
            if self.args.dry_run:
                print "Dry run mode enabled"
            else:
                self._conn().delete_keys(to_del, quiet=not verbose)
            print "Deleted", len(to_del), "files from", self._s3_url()


def main():
    arg_parser = argparse.ArgumentParser(description='s3-mds simple tool')
    arg_parser.add_argument("--action", type=str, help="get|put|copy|info|del|list|clean", required=True)
    arg_parser.add_argument("--bucket", type=str, help="destination bucket", required=True)
    arg_parser.add_argument("--prefix", type=str, help="s3 prefix", required=True)
    arg_parser.add_argument("--filepath", type=str, help="filepath for 'get' or 'put'")
    arg_parser.add_argument("--to", type=str, help="s3 destination prefix for 'copy'")
    arg_parser.add_argument("--keep", type=int, help="keep time in days", default=0)
    arg_parser.add_argument("--limit", type=int, help="limit for 'list' or 'clean' (0 - scan only)", default=0)
    arg_parser.add_argument("--attempts", type=int, help="attempts on 'put'", default=3)
    arg_parser.add_argument("--verbose", default=False, help="verbose mode", action="store_true")
    arg_parser.add_argument("--dry-run", default=False, dest="dry_run", help="dry run mode", action="store_true")
    arg_parser.add_argument(
        "--incremental-clean", default=False, dest="incremental_clean",
        help="enable smart limit for the analysis of objects", action="store_true")

    s3_proxy = S3Proxy(arg_parser.parse_args())
    s3_proxy.process_action()

if __name__ == "__main__":
    main()
