#include <maps/wikimap/mapspro/tools/signals_graph/lib/load_signals_from_yt.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/link.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/generate_edges_from_tracks.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/hypotheses_generator/generate_hyps_from_edges.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/generator_arguments.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/generator_parameters.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/read_signals.h>
#include <maps/wikimap/mapspro/tools/signals_graph/lib/generate_hypotheses.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/cmdline/include/cmdline.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>

#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/geom_utils.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/common/config.h>

#include <string>
#include <fstream>
#include <memory>

namespace signals_graph = maps::wiki::signals_graph;
namespace common = maps::wiki::common;

namespace {

constexpr int MAX_SIGNALS_LOAD = 100 * 1000 * 1000;

maps::geolib3::BoundingBox bufferBbox(
    const maps::geolib3::BoundingBox& bbox, double bufferMeters) {

    auto lower = maps::geolib3::geoPoint2Mercator(bbox.lowerCorner());
    auto upper = maps::geolib3::geoPoint2Mercator(bbox.upperCorner());
    const double mercatorDistance = maps::geolib3::toMercatorUnits(bufferMeters, bbox.center());

    maps::geolib3::Vector2 diagonal{mercatorDistance, mercatorDistance};

    lower -= diagonal;
    upper += diagonal;

    return {
        maps::geolib3::mercator2GeoPoint(lower),
        maps::geolib3::mercator2GeoPoint(upper)
    };
}

} // namespace

int main(int argc, char** argv) try {
    NYT::Initialize(argc, const_cast<const char**>(argv));

    maps::log8::setLevel(maps::log8::Level::INFO);

    maps::cmdline::Parser parser;

    parser.section("Common options");
    auto prefix = parser.string("prefix").required().help(
        "Prefix for intermediate and result file names");
    auto parametersFile = parser.string("params-file").required().help(
        "Protobuf file with inner algorithm parameters");
    auto argumentsFile = parser.string("args-file").required().help(
        "Protobuf config file with user arguments");

    parser.section("Tracks to edges config");
    auto localGpsTracksFilename = parser.string("gps-file").help(
        "Use this file with signals instead of loading from YT");
    auto tmpTableSuffix = parser.string("tmp-table-suffix").defaultValue("tmp").help(
        "Unique suffix for temporary table for concurrent launches");


    parser.section("Edges to hypotheses config");
    auto servicesConfigFile = parser.string("services-config")
        .help("Mapspro services config");
    auto commitId = parser.size_t("commit-id").defaultValue(0).help(
        "Revision commit ID");

    parser.section("Evaluation options");
    auto calculateMetric = parser.flag("metric").defaultValue(false).help(
        "Calculate completeness metric. Hypotheses won't be generated");
    auto renderGeoms = parser.flag("render").defaultValue(false).help(
        "Render difference output");

    parser.parse(argc, argv);

    std::unique_ptr<common::ExtendedXmlDoc> configXml;
    if (servicesConfigFile.defined()) {
        configXml = std::make_unique<common::ExtendedXmlDoc>(servicesConfigFile);
    } else {
        configXml = common::loadDefaultConfig();
    }

    std::string filenamePrefix = prefix + '_';

    signals_graph::GeneratorParameters genParams(parametersFile);
    signals_graph::GeneratorArguments genArgs(argumentsFile);

    std::vector<signals_graph::Signal> signals;
    if (localGpsTracksFilename.defined()) {
        signals = signals_graph::readSignals(localGpsTracksFilename, genArgs.maxSignals());
    } else {
        auto bufferedGeoBbox = bufferBbox(
            genArgs.geoBbox(),
            genParams.signalsLoadBufferMeters()
        );

        signals = signals_graph::loadSignalsFromYt(
            bufferedGeoBbox,
            genArgs.datePeriod(),
            MAX_SIGNALS_LOAD,
            tmpTableSuffix
        );
    }

    auto hypotheses = generateHypotheses(
        signals,
        filenamePrefix,
        genParams,
        genArgs,
        *configXml,
        renderGeoms,
        calculateMetric,
        commitId
    );

    INFO() << "Output " << hypotheses.size() << " hypotheses";
    signals_graph::generateHypothesesFile(filenamePrefix + "hypotheses.tsv", hypotheses);

    return 0;
} catch (maps::Exception& ex) {
    ERROR() << "Caught maps::Exception: " << ex.what();
} catch (std::exception& ex) {
    ERROR() << "Caught std::exception: " << ex.what();
}
