#pragma once

#include <maps/wikimap/mapspro/tools/signals_graph/proto/missing_roads_arguments.pb.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/point.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/config/include/config.h>

#include <google/protobuf/text_format.h>

#include <util/generic/string.h>

#include <boost/date_time/gregorian/gregorian.hpp>

#include <string>
#include <fstream>

namespace maps {
namespace wiki {
namespace signals_graph {

class GeneratorArguments {
public:
    /**
     * @param filename Path to file with config in protobuf text format
     */
    explicit GeneratorArguments(const std::string& filename) {
        TString protoString;
        protoString.assign(maps::config::readConfigFile(filename).c_str());
        MissingRoadsConfig conf;
        REQUIRE(
            google::protobuf::TextFormat::ParseFromString(protoString, &conf),
            "Could not parse arguments file " << filename
        );

        auto bbox = conf.GetBBox();

        geoBbox_ = geolib3::BoundingBox{
            {bbox.GetLonMin(), bbox.GetLatMin()},
            {bbox.GetLonMax(), bbox.GetLatMax()}
        };

        startDate_ = conf.GetSignalsStartDate();
        endDate_ = conf.GetSignalsEndDate();
        maxSignals_ = conf.GetSignalsMaxCount();
    }

    GeneratorArguments(
        const geolib3::BoundingBox& geoBbox,
        const std::string& startDate,
        const std::string& endDate,
        size_t maxSignals)
        : geoBbox_(geoBbox)
        , startDate_(startDate)
        , endDate_(endDate)
        , maxSignals_(maxSignals)
    {}

    const geolib3::BoundingBox& geoBbox() const { return geoBbox_; }

    boost::gregorian::date_period datePeriod() const {
        return {
            boost::gregorian::from_simple_string(startDate_),
            boost::gregorian::from_simple_string(endDate_)
        };
    }
    size_t maxSignals() const { return maxSignals_; }

private:
    geolib3::BoundingBox geoBbox_; // {lon, lat}, {lon, lat}
    std::string startDate_;
    std::string endDate_;
    size_t maxSignals_;
};

} // namespace signals_graph
} // namespace wiki
} // namespace maps
