#include "grid.h"

#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <yandex/maps/wiki/common/extended_xml_doc.h>

#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/vector.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/geolib/include/conversion_geos.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

namespace rf = maps::wiki::revision::filters;

const std::string CAT_RD_EL = "cat:rd_el";

namespace maps {
namespace wiki {
namespace signals_graph {

namespace {

revision::Revisions
getRoadRevisionsInBbox(
    const wiki::revision::Snapshot& snapshot,
    const maps::geolib3::BoundingBox& bbox
) {
    // Can limit FunClass with `&& rf::Attr("rd_el:fc") < 7`
    rf::ProxyFilterExpr roadElementFilter =
        rf::ObjRevAttr::isNotDeleted() &&
            rf::ObjRevAttr::isNotRelation() &&
            rf::Geom::defined() &&
            rf::Attr(CAT_RD_EL).defined();

    auto pointAreaFilter = rf::GeomFilterExpr(
        rf::GeomFilterExpr::Operation::IntersectsLinestrings,
        bbox.minX(),
        bbox.minY(),
        bbox.maxX(),
        bbox.maxY()
    );

    auto filter = roadElementFilter && pointAreaFilter;

    return snapshot.objectRevisionsByFilter(filter);
}

std::vector<geolib3::Polyline2> extractGeometries(const revision::Revisions& revisions) {
    std::vector<geolib3::Polyline2> result;

    for (const auto& road : revisions) {
        if (!road.data().geometry) {
            continue;
        }

        std::istringstream geomStream(*road.data().geometry);
        result.push_back(geolib3::WKB::read<geolib3::Polyline2>(geomStream));
    }

    return result;
}

} // namespace

class GatewayHandler {
public:
    GatewayHandler(const common::ExtendedXmlDoc& config, std::string poolName)
        : poolHolder(config, poolName, "long-read")
        , transactionHandle(poolHolder.pool().slaveTransaction())
        , revisionsGateway(transactionHandle.get()) {}

    const revision::RevisionsGateway& getGateway() { return revisionsGateway; }

    revision::Snapshot getLastSnapshot() {
        INFO() << "Last snapshot commit id:" << revisionsGateway.maxSnapshotId().commitId();
        return revisionsGateway.snapshot(revisionsGateway.maxSnapshotId());
    }

    revision::Snapshot getSnapshot(revision::DBID commitId) {
        return revisionsGateway.snapshot(commitId);
    }

private:
    common::PoolHolder poolHolder;
    pgpool3::TransactionHandle transactionHandle;
    revision::RevisionsGateway revisionsGateway;
};

std::vector<geolib3::Polyline2> getGeometriesFromDB(
    const common::ExtendedXmlDoc& config,
    const std::string& base,
    const geolib3::BoundingBox& bbox,
    revision::DBID commitId
) {
    GatewayHandler gateway(config, base);
    auto snapshot = gateway.getLastSnapshot();

    if (commitId != 0) {
        snapshot = gateway.getSnapshot(commitId);
    }

    auto revisions = getRoadRevisionsInBbox(snapshot, bbox);

    return extractGeometries(revisions);
}

} // namespace signals_graph
} // namespace wiki
} // namespace maps
