#include "generate_hyps_from_edges.h"

#include "buffer_difference.h"
#include "../edge.h"
#include "edge_hypothesis.h"
#include "editor_interaction.h"
#include "grid.h"
#include "metrics.h"
#include "../save.h"

#include <yandex/maps/wiki/common/extended_xml_doc.h>

#include <maps/libs/log8/include/log8.h>

#include <fstream>

namespace maps {
namespace wiki {
namespace signals_graph {
namespace {

void renderInputGeometries(
    const std::vector<geolib3::Polyline2>& revisionEdges,
    const std::vector<Edge>& edges,
    const std::string& output_prefix
) {
    std::ofstream initialGeomFile(output_prefix + "all.txt");
    drawEdges(initialGeomFile, edges);
    renderGeometriesEasyView(initialGeomFile, revisionEdges, GREEN);
}

void renderDifferenceResult(
    const std::vector<common::Geom>& resultGeometries,
    const std::vector<Edge>& edges,
    const std::string& output_prefix,
    bool reverse = false
) {

    std::string result_file_suffix = reverse ? "rev_res.txt" : "res.txt";
    std::ofstream diffFile(output_prefix + result_file_suffix);

    printLinestyle(RED, diffFile);
    for (size_t i = 0; i < resultGeometries.size(); ++i) {
        const auto& edge = edges[i];
        std::stringstream description;
        description << "Edge_" << edge.weight << "_from" << edge.from << "_to" << edge.to;

        auto geom = resultGeometries[i].geosGeometryPtr();
        renderEasyView(
            diffFile,
            geom,
            geom->getGeometryTypeId(),
            description.str());
    }
}

/**@param geometries Parts of hypotheses, not present in revision graph
 * @param edges Initial hypotheses and their geometries
 *
 * Parameters @p edges and @p geometries should be aligned with each other.
 */
std::vector<EdgeHypothesis> createHypotheses(
    std::vector<common::Geom> geometries,
    std::vector<Edge> edges
) {
    std::vector<EdgeHypothesis> result;
    result.reserve(geometries.size());

    for (size_t i = 0; i < geometries.size(); ++i) {
        if (geometries[i].isNull()) {
            continue;
        }
        auto typeId = geometries[i].geosGeometryPtr()->getGeometryTypeId();
        if (typeId != geos::geom::GEOS_LINESTRING && typeId != geos::geom::GEOS_MULTILINESTRING) {
            WARN() << "Got wrong hypothesis type: " << geometries[i]->getGeometryType();
            continue;
        }

        result.emplace_back(geometries[i], edges[i].weight);
    }
    return result;
}

std::ostream& operator<<(std::ostream& ostr, const EdgeHypothesis& hypothesis) {
    geolib3::Point2 geoPoint = geolib3::mercator2GeoPoint(hypothesis.point);
    ostr << std::setprecision(9) << geoPoint.x() << '\t' << geoPoint.y() << '\t';
    ostr << "Missing road";
    ostr << " [W=" << hypothesis.weight << ";L=" << hypothesis.length << "]";
    return ostr << std::endl;
}

} // namespace

void generateHypothesesFile(
    const std::string& filename,
    std::vector<EdgeHypothesis> hypotheses
) {
    std::ofstream hypothesesFile(filename);
    hypothesesFile << "lon\tlat\tdescription\n";
    for (const auto& hypothesis : hypotheses) {
        hypothesesFile << hypothesis;
    }
}

std::vector<EdgeHypothesis> generateHypothesesFromEdges(
    const std::vector<Edge>& edges,
    double bufferWidth,
    double graphComparisonCellSpan,
    const geolib3::BoundingBox& bbox,
    const common::ExtendedXmlDoc& configXml,
    const std::string& filenamePrefix,
    bool renderGeoms,
    bool calculateMetric,
    revision::DBID commitId
) {
    INFO() << "Reading graph from editor...";

    auto revisionEdges = getGeometriesFromDB(configXml, "long-read", bbox, commitId);
    Grid grid(bbox, graphComparisonCellSpan);

    std::vector<geolib3::Polyline2> gpsEdges;
    for (const Edge& edge: edges) {
        gpsEdges.emplace_back(edge.polyline);
    }

    INFO() << "Calculating buffer difference...";
    DifferenceResult result;
    std::vector<EdgeHypothesis> hypotheses;
    if (!calculateMetric) {
        result = geomBufferDifference(gpsEdges, revisionEdges, grid, bufferWidth);

        INFO() << "Got " << result.geometries.size() << " diff geometries";

        hypotheses = createHypotheses(result.geometries, edges);
    } else {
        result = geomBufferDifference(revisionEdges, gpsEdges, grid, bufferWidth);

        auto metrics = calculateMetrics(revisionEdges, edges, result.geometries, grid);
        printMetrics(metrics);
    }

    if (renderGeoms) {
        INFO() << "Rendering for EasyView";
        renderInputGeometries(revisionEdges, edges, filenamePrefix);
        renderDifferenceResult(result.geometries, edges, filenamePrefix, calculateMetric);
    }

    return hypotheses;
}

} // namespace signals_graph
} // namespace wiki
} // namespace maps
