#include <maps/wikimap/mapspro/tools/signals_graph/proto/table_entry.pb.h>

#include "signal.h"

#include <yandex/maps/wiki/common/yt.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/vector.h>

#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/chrono/include/days.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <mapreduce/yt/interface/client.h>
#include <util/generic/guid.h>

#include <boost/date_time/gregorian/gregorian.hpp>

#include <string>
#include <fstream>
#include <iostream>

using namespace maps::chrono::literals;

namespace maps {
namespace wiki {
namespace signals_graph {
namespace {

namespace bgreg = boost::gregorian;

const TString YT_PROXY_NAME = "hahn";

const std::string SIGNALS_TABLES_DIR = "//home/maps/jams/production/data/signals/";

const std::string OPERATION_URL = "https://yt.yandex-team.ru/hahn/?page=operation&mode=detail&id=";

std::vector<std::string> generateTableNames(bgreg::date_period datePeriod) {
    std::vector<std::string> result;
    for (auto date = datePeriod.begin();
        date != datePeriod.end();
        date += bgreg::days(1))
    {
        result.emplace_back(SIGNALS_TABLES_DIR + bgreg::to_iso_extended_string(date));
    }

    return result;
}

bool inAreaOfInterest(const TableEntry& row,
                      double minLon, double maxLon, double minLat, double maxLat)
{
    return minLat <= row.lat() && row.lat() <= maxLat &&
        minLon <= row.lon() && row.lon() <= maxLon;
}

class FilterSignals : public NYT::IMapper<
    NYT::TTableReader<TableEntry>, NYT::TTableWriter<TableEntry>>
{
public:
    Y_SAVELOAD_JOB(
        minLon_,
        maxLon_,
        minLat_,
        maxLat_
    );

    FilterSignals() = default;

    FilterSignals(
        double minLon, double maxLon, double minLat, double maxLat
    ) : minLon_(minLon), maxLon_(maxLon), minLat_(minLat), maxLat_(maxLat) {}

    void Do(TReader *reader, TWriter *writer) override
    {
        for (; reader->IsValid(); reader->Next()) {
            const auto& row = reader->GetRow();

            if (inAreaOfInterest(row, minLon_, maxLon_, minLat_, maxLat_)) {
                writer->AddRow(row);
            }
        }
    }

private:
    double minLon_;
    double maxLon_;
    double minLat_;
    double maxLat_;
};

REGISTER_MAPPER(FilterSignals);

void waitForOperation(NYT::IOperationPtr operation) {
    INFO() << "Operation page: " << OPERATION_URL <<
        GetGuidAsString(operation->GetId());
    auto operationComplete = operation->Watch();
    operationComplete.Wait();
    operationComplete.GetValue();
}

} // namespace

std::vector<Signal> loadSignalsFromYt(
    const geolib3::BoundingBox& geoBbox,
    boost::gregorian::date_period datePeriod,
    int maxSignalsNumber,
    const std::string& tmpTableSuffix
) {
    NYT::IClientPtr client = common::yt::createYtClient(YT_PROXY_NAME);
    auto transaction = client->StartTransaction();

    INFO() << "Loading signals from YT";

    const TString tmpTable("//tmp/tracks_load_" + tmpTableSuffix);
    auto mapOperationSpec = NYT::TMapOperationSpec()
        .AddOutput<TableEntry>(tmpTable);

    auto inputTables = generateTableNames(datePeriod);
    for (const auto& table : inputTables) {
        if (transaction->Exists(TString(table))) {
            mapOperationSpec.AddInput<TableEntry>(
                NYT::TRichYPath(table.c_str()).Columns({
                    "lon", "lat", "direction", "speed", "timestamp", "uuid"})
            );
        } else {
            WARN() << "Signals table " << table << " doesn't exist on YT";
        }
    }

    REQUIRE(
        !mapOperationSpec.Inputs_.empty(),
        "No valid signals tables"
    );

    INFO() << "Begin filter signals...";

    auto mapOperation = transaction->Map(
        mapOperationSpec,
        new FilterSignals(geoBbox.minX(), geoBbox.maxX(), geoBbox.minY(), geoBbox.maxY()),
        NYT::TOperationOptions().Wait(false)
    );
    waitForOperation(mapOperation);

    INFO() << "Count...";

    auto signalsNumber = transaction->Get(tmpTable + "/@row_count").AsInt64();
    if (signalsNumber > maxSignalsNumber) {
        WARN() << "Too many signals acquired (" << signalsNumber << ")." <<
            "Will load only " << maxSignalsNumber;
    }

    const TString outTable("//tmp/tracks_load_sort_" + tmpTableSuffix);

    INFO() << "Sort signals...";

    auto sortOperation = transaction->Sort(
        NYT::TSortOperationSpec()
            .AddInput(tmpTable)
            .SortBy({"uuid", "timestamp"})
            .Output(outTable),
        NYT::TOperationOptions().Wait(false)
    );
    waitForOperation(sortOperation);

    std::vector<Signal> signals;
    auto reader = transaction->CreateTableReader<TableEntry>(outTable);
    for (; reader->IsValid() && maxSignalsNumber--; reader->Next()) {
        const auto& row = reader->GetRow();
        signals.push_back({
            { row.lon(), row.lat() },
            row.direction(),
            row.speed(),
            row.timestamp(),
            row.uuid()
        });
    }

    // Don't commit temporary tables
    transaction->Abort();

    INFO() << "Signals loaded";

    return signals;
}

} // namespace signals_graph
} // namespace wiki
} // namespace maps
