#include "save.h"

#include "link.h"
#include "style.h"

#include <string>
#include <vector>
#include <fstream>
#include <iomanip>

namespace maps {
namespace wiki {
namespace signals_graph {

std::ostream& printPoint(std::ostream& ostr, maps::geolib3::Point2 point) {
    return ostr << std::setprecision(9) << point.x() << " " << point.y();
}

void drawTurns(const std::vector<Turn>& turns, std::ofstream& ostr) {
    printPointstyle(RED, RED, ostr, 5);

    for (size_t i = 0; i < turns.size(); ++i) {
        const auto& turn = turns[i];

        printPoint(ostr, turn.mercPoint) << " " << turn.uuid << "_point_" <<
            "#" << i << "[" <<
            turn.averageSpeed << "]s_[" <<
            turn.directionIn.angle() << "]in_[" <<
            turn.directionOut.angle() << "]out_[" <<
            turn.angle << "]diff_[" <<
            turn.weight << "]w" <<
            std::endl;
    }
}

void drawLinks(
    const std::vector<Link>& links,
    std::ofstream& ostr,
    const std::vector<Track>& tracks
) {
    printLinestyle(RED, ostr);

    for (const auto& edge : links) {
        const std::vector<Signal>& points = tracks[edge.trackId].points();

        for (size_t pos = edge.posFrom; pos <= edge.posTo; ++pos) {
            printPoint(ostr, points[pos].geoPoint) << " ";
        }
        ostr << tracks[edge.trackId].uuid() << "_" << edge.crossFrom << "-" << edge.crossTo << "_edge" << std::endl;
    }
}

void drawEdge(std::ostream& ostr, const Edge& edge) {
    for (size_t i = 0; i < edge.polyline.pointsNumber(); ++i) {
        printPoint(ostr, edge.polyline.pointAt(i)) << " ";
    }
    ostr << "edge_" << edge.weight
        << "_from" << edge.from << "_to" << edge.to;
    ostr << std::endl;
}

void drawEdges(std::ostream& ostr, const std::vector<Edge>& edges) {
    printLinestyle(RED, ostr);

    for (const Edge& edge : edges) {
        drawEdge(ostr, edge);
    }
}

std::ostream& saveLink(
    std::ostream& ostr, const Link& link, const std::vector<Track>& tracks
) {
    ostr << 1 /* weight */ << " " << link.crossFrom << " " << link.crossTo << " ";

    ostr << link.posTo - link.posFrom + 1 << " ";
    for(size_t i = link.posFrom; i <= link.posTo; ++i) {
        const auto& point = tracks[link.trackId].points()[i].geoPoint;
        printPoint(ostr, geoPoint2Mercator(point)) << " ";
    }

    return ostr << std::endl;
}

void saveEdge(std::ostream& ostr, const Edge& edge) {
    ostr << edge.weight << " " << edge.from << " " << edge.to << " ";
    ostr << edge.polyline.pointsNumber() << " ";
    for (size_t i = 0; i < edge.polyline.pointsNumber(); ++i) {
        printPoint(ostr, edge.polyline.pointAt(i)) << " ";
    }
    ostr << std::endl;
}

Edge loadEdge(std::istream& istr) {
    size_t weight;
    int from;
    int to;
    istr >> weight >> from >> to;

    size_t pointsNumber;
    istr >> pointsNumber;

    std::vector<geolib3::Point2> points;
    points.reserve(pointsNumber);
    for (size_t i = 0; i < pointsNumber; ++i) {
        double x;
        double y;
        istr >> x >> y;
        points.emplace_back(x, y);
    }

    return Edge(points, weight, from, to);
}

void saveData(
    const std::vector<Track>& tracks,
    const std::vector<Link>& links,
    const std::vector<Edge>& edges,
    std::string linksFilename,
    std::string edgesFilename
) {
    std::ofstream linksFile(linksFilename);
    linksFile << links.size() << std::endl;
    for (const auto& link : links) {
        saveLink(linksFile, link, tracks);
    }

    std::ofstream edgesFile(edgesFilename);
    edgesFile << edges.size() << std::endl;
    for (const auto& edge : edges) {
        saveEdge(edgesFile, edge);
    }
}

} // namespace maps
} // namespace wiki
} // namespace signals_graph
