#include "turn_visitor.h"

namespace maps {
namespace wiki {
namespace signals_graph {

constexpr double MAX_TURN_SPEED_MPS = 50.0 / 3.6;
constexpr double MIN_ANGLE_DIFF_RAD = M_PI / 12;

void visitTracks(std::vector<Track>& tracks, Visitor& visitor) {
    for (Track& track : tracks) {
        visitor.visitTrack(track);
    }
}

Turn createTurn(const Signal& prev, const Signal& cur, const Signal& next) {
    geolib3::Polyline2 track{
        {
            prev.mercPoint(),
            cur.mercPoint(),
            next.mercPoint()
        }
    };

    Turn result;

    result.uuid = cur.uuid;

    result.mercPoint = track.pointAt(1);

    double length = geolib3::length(track.segmentAt(0));
    double duration = cur.timestamp - prev.timestamp;
    result.averageSpeed = length / duration;

    result.directionIn = geolib3::Direction2(track.segmentAt(0));
    result.directionOut = geolib3::Direction2(track.segmentAt(1));
    result.angle = geolib3::angle(
        result.directionIn,
        result.directionOut
    );

    return result;
}

void TurnVisitor::visitTrack(Track& track) {
    const auto& points = track.points();

    if (points.size() < 2) {
        return;
    }

    for (size_t i = 1; i + 1 < points.size(); ++i) {
        Turn turn = createTurn(points[i-1], points[i], points[i+1]);
        if (validTurn(turn)) {
            validTurns_.push_back(turn);
            track.registerTurn(i, validTurns_.size() - 1);
        }
    }
}

bool TurnVisitor::validTurn(const Turn& turn) {
    return turn.averageSpeed < MAX_TURN_SPEED_MPS &&
        turn.angle > MIN_ANGLE_DIFF_RAD;
}

} // namespace maps
} // namespace wiki
} // namespace signals_graph
