#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/chrono/include/time_point.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/batch.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/default_config.h>
#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/pgpool3_helpers.h>

#include <maps/wikimap/mapspro/libs/configs_editor/include/yandex/maps/wiki/configs/editor/config_holder.h>
#include <maps/wikimap/mapspro/libs/configs_editor/include/yandex/maps/wiki/configs/editor/category_groups.h>

#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/commit.h>

#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/common.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/gateway.h>

#include <chrono>
#include <iomanip>
#include <map>
#include <memory>

namespace maps::wiki::social {
namespace {

const std::string CAT_GROUP_GROUPEDITS = "groupedits_group";
const std::string CAT_GROUP_SERVICE = "service_group";
const std::string NEW_OBJECT_ACTION = "object-created";
const std::string GROUPEDIT_ACTION_PREFIX = "group";
const auto SOCIAL_EVENTS_BATCH_SIZE = 1000;

std::map<TId, chrono::TimePoint> loadCommitIdCreatedAtMap(
    pqxx::transaction_base& txnCore,
    TUid uid)
{
    std::map<TId, chrono::TimePoint> result;
    auto commits = revision::Commit::load(
        txnCore, revision::filters::CommitAttr("created_by") == uid);
    for (const auto& commit : commits) {
        result.emplace(
            commit.id(),
            chrono::parseSqlDateTime(commit.createdAt()));
    }
    return result;
}

struct CategoryGroupStat
{
    size_t totalCount;
    size_t createdCount;
};

using CategoryGroupStats = std::map<std::string, CategoryGroupStat>;

CategoryGroupStats loadStatCategories(
    pqxx::transaction_base& txnSocial,
    TUid uid)
{
    CategoryGroupStats result;
    auto query = "SELECT * FROM social.cat_group_edits_stat WHERE uid=" + std::to_string(uid);
    for (const auto& row : txnSocial.exec(query)) {
        result.emplace(
            row["category_group"].as<std::string>(),
            CategoryGroupStat{row["total_count"].as<size_t>(), row["created_count"].as<size_t>()}
        );
    }
    return result;
}

CategoryGroupStats loadRealStatCategories(
    const configs::editor::ConfigHolder& editorConfig,
    pqxx::transaction_base& txnSocial,
    const TIds& commitIds)
{
    const auto& categoryGroups = editorConfig.categoryGroups();

    CategoryGroupStats result;

    social::Gateway gtw(txnSocial);

    common::applyBatchOp<TIds>(
        commitIds,
        SOCIAL_EVENTS_BATCH_SIZE,
        [&](const auto& batchIds) {
            const auto events = gtw.loadTrunkEditEventsByCommitIds(batchIds);
            for (const auto& event : events) {
                if (auto category = event.getPrimaryObjectCategory(); category) {
                    auto group = categoryGroups.findGroupByCategoryId(category.value());
                    if (!group) {
                        continue;
                    }
                    auto& stat = result[group->id()];
                    stat.totalCount++;
                    if (event.action() == NEW_OBJECT_ACTION) {
                        stat.createdCount++;
                    }
                } else if (event.action().starts_with(GROUPEDIT_ACTION_PREFIX)) {
                    result[CAT_GROUP_GROUPEDITS].totalCount++;
                }
            }
        });
    return result;
}

size_t dumpStatCategories(const CategoryGroupStats& statCategories)
{
    size_t totalCount = 0, createdCount = 0;
    for (const auto& [cat, stat] : statCategories) {
        totalCount += stat.totalCount;
        createdCount += stat.createdCount;
    }
    INFO() << std::setw(30) << "TOTAL" << " : " << totalCount << " : " << createdCount;
    auto srvIt = statCategories.find(CAT_GROUP_SERVICE);
    auto srvCount = srvIt == statCategories.end() ? 0 : srvIt->second.totalCount;
    auto countV2 = totalCount - srvCount;
    INFO() << std::setw(30) << "TOTAL V2" << " : " << countV2;
    return countV2;
};

std::string dumpDiffStatCategories(
    TUid uid, const CategoryGroupStats& old, const CategoryGroupStats& real)
{
    struct Diff { CategoryGroupStat old; CategoryGroupStat real; };
    std::map<std::string, Diff> data;
    for (const auto& [cat, stat] : old) {
        data[cat].old = stat;
    }
    for (const auto& [cat, stat] : real) {
        data[cat].real = stat;
    }

    std::string query;
    for (const auto& [cat, diff] : data) {
        auto isDiff =
            diff.old.totalCount != diff.real.totalCount ||
            diff.old.createdCount != diff.real.createdCount;

        INFO()
            << std::setw(30) << cat << " : "
            << diff.old.totalCount << " : " << diff.old.createdCount << " : "
            << diff.real.totalCount << " : " << diff.real.createdCount << (isDiff ? " !!!" : "");
        if (!isDiff) {
            continue;
        }
        if (!diff.real.totalCount && !diff.real.createdCount) {
            query +=
                "DELETE FROM social.cat_group_edits_stat"
                " WHERE uid = " + std::to_string(uid) +
                    " AND category_group = '" + cat + "';\n";
            continue;
        }
        if (!diff.old.totalCount && !diff.old.createdCount) {
            query +=
                "INSERT INTO social.cat_group_edits_stat"
                " (uid, category_group, total_count, created_count) VALUES (" +
                std::to_string(uid) + ", "
                "'" + cat + "', " +
                std::to_string(diff.real.totalCount) + ", " +
                std::to_string(diff.real.createdCount) +
                ");\n";
            continue;
        }

        std::string updateQuery;
        if (diff.old.totalCount != diff.real.totalCount) {
            updateQuery += "total_count = " + std::to_string(diff.real.totalCount);
        }
        if (diff.old.createdCount != diff.real.createdCount) {
            updateQuery += updateQuery.empty() ? "" : ", ";
            updateQuery += "created_count = " + std::to_string(diff.real.createdCount);
        }
        if (!updateQuery.empty()) {
            query +=
                "UPDATE social.cat_group_edits_stat SET " + updateQuery +
                " WHERE uid = " + std::to_string(uid) +
                    " AND category_group = '" + cat + "';\n";
        }
    }
    return query;
}

struct Stats
{
    Stats() = default;

    explicit Stats(const pqxx::row& row)
    {
        total = row["total_edits"].as<size_t>(0);
        totalV2 = row["total_edits_v2"].as<size_t>(0);
        for (size_t ago : { 1, 7, 30, 90 }) {
            agoCounts[ago] = row["edits_" + std::to_string(ago) + "d_ago_v2"].as<size_t>(0);
        }
    }

    size_t total = 0;
    size_t totalV2 = 0;
    std::map<size_t, size_t> agoCounts;

    void dump() const
    {
        INFO()
            << " total_edits: " << total << "\n"
            << " total_edits_v2: " << totalV2 << "\n"
            << " edits_1d_ago_v2: " << agoCounts.at(1) << "\n"
            << " edits_7d_ago_v2: " << agoCounts.at(7) << "\n"
            << " edits_30d_ago_v2: " << agoCounts.at(30) << "\n"
            << " edits_90d_ago_v2: " << agoCounts.at(90);
    }

    std::string makeUpdateQuery(TUid uid, const Stats& realStats) const
    {
        REQUIRE(
            total == realStats.total,
            "Not equal total edits, old: " << total << " real: " << realStats.total);

        std::string query;
        if (totalV2 != realStats.totalV2) {
            query += "total_edits_v2 = " + std::to_string(realStats.totalV2);
        }
        for (auto days : { 1, 7, 30, 90 }) {
            auto counter = realStats.agoCounts.at(days);
            if (agoCounts.at(days) != counter) {
                query += query.empty() ? "" : ", ";
                query += "edits_" + std::to_string(days) + "d_ago_v2 = " + std::to_string(counter);
            }
        }
        if (query.empty()) {
            return {};
        }
        return
            "UPDATE social.stats SET " + query +
            " WHERE uid = " + std::to_string(uid) + ";\n";
    }
};

void run(
    const common::ExtendedXmlDoc& config,
    TUid uid,
    const std::string& login,
    bool fix,
    bool dryRun)
{
    configs::editor::ConfigHolder editorConfig(
        config.get<std::string>("/config/services/editor/config"));

    common::PoolHolder core(config, "core", "grinder");
    common::PoolHolder social(config, "social", "grinder");

    auto txnCore = core.pool().masterReadOnlyTransaction();
    if (!login.empty() && !uid) {
        uid = acl::ACLGateway(*txnCore).user(login).uid();
        INFO() << "Uid: " << uid;
    }

    auto txnSocial = fix
        ? social.pool().masterWriteableTransaction()
        : social.pool().masterReadOnlyTransaction();
    auto rows = txnSocial->exec(
        "SELECT * FROM social.stats WHERE uid=" + std::to_string(uid));
    if (rows.empty()) {
        INFO() << "Stats: empty";
        return;
    }

    auto commitIdCreatedAtMap = loadCommitIdCreatedAtMap(*txnCore, uid);
    INFO() << "Commits count: " << commitIdCreatedAtMap.size();


    Stats stats(rows[0]);

    TIds allCommitIds;
    for (const auto& [commitId, createdAt] : commitIdCreatedAtMap) {
        allCommitIds.emplace(commitId);
    }

    auto getCommitIds = [&] (chrono::TimePoint before) {
        TIds result;
        for (const auto& [commitId, createdAt] : commitIdCreatedAtMap) {
            if (allCommitIds.contains(commitId) && createdAt < before) {
                result.emplace(commitId);
            }
        }
        return result;
    };

    auto add = [](CategoryGroupStats& stats1, const CategoryGroupStats& stats2) {
        for (const auto& [cat, stat] : stats2) {
            auto& st = stats1[cat];
            st.totalCount += stat.totalCount;
            st.createdCount += stat.createdCount;
        }
    };

    Stats realStats;
    realStats.total = commitIdCreatedAtMap.size();

    auto now = std::chrono::system_clock::now();
    CategoryGroupStats groupStatsAgo;

    for (size_t days : { 90, 30, 7, 1 }) {
        INFO() << "=== " << days << "d ago ===";
        auto commitIds = getCommitIds(now - std::chrono::days(days));
        auto realStatCategories = loadRealStatCategories(editorConfig, *txnSocial, commitIds);
        add(groupStatsAgo, realStatCategories);
        realStats.agoCounts[days] = dumpStatCategories(groupStatsAgo);

        for (auto id : commitIds) {
            allCommitIds.erase(id);
        }
    }

    INFO() << "=== OLD ===";
    auto oldStatCategories = loadStatCategories(*txnSocial, uid);
    auto oldTotalV2 = dumpStatCategories(oldStatCategories);
    INFO() << "total_v2: " << oldTotalV2;
    stats.dump();

    INFO() << "=== REAL ===";
    auto realStatCategories = loadRealStatCategories(editorConfig, *txnSocial, allCommitIds);
    add(realStatCategories, groupStatsAgo);
    realStats.totalV2 = dumpStatCategories(realStatCategories);
    realStats.dump();

    INFO() << "=== DIFF old vs real ===";
    auto query =
        dumpDiffStatCategories(uid, oldStatCategories, realStatCategories) +
        stats.makeUpdateQuery(uid, realStats);
    if (query.empty()) {
        return;
    }
    INFO() << "\n" + query;
    if (!fix) {
        return;
    }
    txnSocial->exec(query);
    if (dryRun) {
        INFO() << "DRY RUN MODE: Results not saved";
    } else {
        txnSocial->commit();
        INFO() << "Results saved";
    }
}

} // namespace
} // namespace maps::wiki::social

int main(int argc, char** argv) try {
    maps::cmdline::Parser argsParser;
    auto config = argsParser.file("config")
        .help("Path to config.xml with database connection settings.");
    auto uid = argsParser.size_t("uid")
        .help("Passport uid (puid)");
    auto login = argsParser.string("login")
        .help("Passport login (<login>.yandex.ru)");
    auto fixStat = argsParser
        .flag("fix")
        .help("fix user counters")
        .defaultValue(false);
    auto dryRun = argsParser
        .flag("dry-run")
        .help("do not update data in database")
        .defaultValue(false);
    argsParser.parse(argc, argv);

    if (!uid.defined() && !login.defined()) {
        argsParser.printUsageAndExit(EXIT_FAILURE);
    }

    auto configPtr = config.defined()
        ? std::make_unique<maps::wiki::common::ExtendedXmlDoc>(config)
        : maps::wiki::common::loadDefaultConfig();

    maps::wiki::social::run(*configPtr, uid, login, fixStat, dryRun);
    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    FATAL() << "Exception caught: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& ex) {
    FATAL() << "Exception caught: " << ex.what();
    return EXIT_FAILURE;
} catch (...) {
    FATAL() << "Unknown exception caught";
    return EXIT_FAILURE;
}
