#include "cmd_helpers.h"
#include "common.h"

#include <maps/libs/common/include/exception.h>

#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>

#include <iostream>
#include <fstream>

namespace po = boost::program_options;
namespace fixer = maps::wiki::topology_fixer;
namespace cmd = fixer::cmd;

namespace {

std::string readFileToString(const std::string& filePath)
{
    std::ostringstream result;
    if (!filePath.empty()) {
        std::string str;
        std::ifstream f(filePath.c_str());
        while (getline(f, str)) {
            result << str << std::endl;
        }
    }
    return result.str();
}

fixer::DBIdType maxYMapsDFID(
    const std::string& connStr,
    const std::string& schemaName,
    const std::string& maxIdSqlPath)
{
    pqxx::connection conn(connStr);
    pqxx::work work(conn);

    work.exec("SET search_path=" + schemaName + ",public");
    auto result = work.exec(readFileToString(maxIdSqlPath));
    return result.at(0).at(0).as<fixer::DBIdType>();
}

} // namespace

int main(int argc, char* argv[])
{
    po::options_description desc("Allowed options");
    cmd::addOptionDescription(desc,
        cmd::g_connectionOption(),
        cmd::g_schemaOption(),
        cmd::g_helpOption(),
        cmd::g_maxIdSqlPathOption());

    po::variables_map vm;
    po::store(po::parse_command_line(argc, argv, desc), vm);
    po::notify(vm);

    if (argc < 2 || vm.count(cmd::OPT_HELP)) {
        std::cerr << "Usage: last_ymapsdf_id <options> " << std::endl;
        std::cerr << desc << std::endl;
        return 1;
    }
    try {
        std::string connStr = cmd::loadParam(vm, cmd::g_connectionOption());
        std::string ymapsdfSchema = cmd::loadParam(vm, cmd::g_schemaOption());
        std::string maxIdSqlPath = cmd::loadParam(vm, cmd::g_maxIdSqlPathOption());

        std::cout << maxYMapsDFID(connStr, ymapsdfSchema, maxIdSqlPath) << std::endl;

        return 0;
    }
    catch (const std::exception &e) {
        std::cerr << e.what();
        return 1;
    }
    catch (...) {
        std::cerr << "Unknown error";
        return 2;
    }
}
