#pragma once

#include "topology_group.h"
#include "common.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/optional.hpp>

#include <string>
#include <sstream>

namespace maps {
namespace wiki {
namespace topology_fixer {
namespace cmd {

namespace po = boost::program_options;

class MissingParameterException : public maps::Exception {};

template <typename T>
struct CmdParam {
    typedef T type;
    std::string name;
    std::string description;
    boost::optional<T> defaultValue; // if not set, parameter is required
};

template <typename... Ts>
void addOptionDescription(po::options_description& desc, const CmdParam<Ts>&... params);

template <typename T>
void addOptionDescription(po::options_description& desc, const CmdParam<T>& param)
{
    if (param.defaultValue) {
        desc.add_options()(
            param.name.c_str(),
            po::value<typename CmdParam<T>::type>()->default_value(*param.defaultValue),
            param.description.c_str());
    } else {
        desc.add_options()(
            param.name.c_str(),
            po::value<typename CmdParam<T>::type>(),
            param.description.c_str());
    }
}

template <typename T, typename... Ts>
void addOptionDescription(
    po::options_description& desc,
    const CmdParam<T>& param, const CmdParam<Ts>&... params)
{
    addOptionDescription<T>(desc, param);
    addOptionDescription<Ts...>(desc, params...);
}

template <typename T>
T
loadParam(const po::variables_map& vm, const CmdParam<T>& param)
{
    if (param.defaultValue) {
        return vm.count(param.name) ? vm[param.name].template as<T>() : *param.defaultValue;
    }
    if (vm.count(param.name)) {
        return vm[param.name].template as<T>();
    }

    throw MissingParameterException()
        << "Parameter " << param.name << " is required but missing";
}


const std::string OPT_HELP = "help";
const std::string OPT_VERBOSE = "verbose";
const std::string OPT_CONN = "conn";
const std::string OPT_RESTRICTIONS_CONFIG = "restrictions-config";
const std::string OPT_FT_GROUP = "group";
const std::string OPT_SCHEMA = "schema";
const std::string OPT_SRID = "srid";
// topology_fixer
const std::string OPT_INPUT_SCHEMA = "input-schema";
const std::string OPT_OUTPUT_SCHEMA = "output-schema";
const std::string OPT_START_YMAPSDF_ID = "start-ymapsdf-id";
// topology_checker
const std::string OPT_ORIGINAL_SCHEMA = "original-schema";
const std::string OPT_FIXED_SCHEMA = "fixed-schema";
const std::string OPT_BOX_SIZE = "box-size";
const std::string OPT_AREA_RELATIVE_DIFFERENCE = "area-relative-difference";
const std::string OPT_PERIMETER_RELATIVE_DIFFERENCE = "perimeter-relative-difference";
const std::string OPT_THREADS = "threads";
// max_ymapsdf_id
const std::string OPT_MAX_ID_SQL_PATH = "max-id-sql-path";

const std::string DEFAULT_LOG_LEVEL = "info";
const double DEFAULT_BOX_SIZE = 200000.0;
const double DEFAULT_AREA_RELATIVE_DIFFERENCE = 0.01;
const double DEFAULT_PERIMETER_RELATIVE_DIFFERENCE = 0.01;

const std::string YMAPSDF_MAX_ID_SQL_PATH = "/usr/share/yandex/maps/ymapsdf2/max_id.sql";

const size_t DEFAULT_THREADS = 15;

// common options

const CmdParam<std::string>& g_helpOption();
const CmdParam<std::string>& g_verboseOption();
const CmdParam<std::string>& g_connectionOption();
const CmdParam<std::string>& g_restrictionsConfigOption();
const CmdParam<std::string>& g_ftGroupOption();
const CmdParam<size_t>& g_threadsOption();
const CmdParam<std::string>& g_schemaOption();
const CmdParam<std::string>& g_sridOption();

// topology_fixer options

const CmdParam<std::string>& g_inputSchemaOption();
const CmdParam<std::string>& g_outputSchemaOption();
const CmdParam<DBIdType>& g_startYMapsDFIdOption();

// topology_checker options

const CmdParam<std::string>& g_originalSchemaOption();
const CmdParam<std::string>& g_fixedSchemaOption();
const CmdParam<double>& g_boxSizeOption();
const CmdParam<double>& g_areaRelativeDifferenceOption();
const CmdParam<double>& g_perimeterRelativeDifferenceOption();

// max ymapsdf id options

const CmdParam<std::string>& g_maxIdSqlPathOption();

} // namespace cmd
} // namespace topology_fixer
} // namespace wiki
} // namespace maps
