#pragma once

#include <maps/libs/common/include/exception.h>

#include <pqxx/pqxx>

#include <cstdint>
#include <string>
#include <map>
#include <list>
#include <set>
#include <unordered_set>
#include <atomic>
#include <algorithm>

namespace maps {
namespace wiki {
namespace topology_fixer {

typedef int64_t DBIdType;

typedef DBIdType NodeId;
typedef DBIdType EdgeId;
typedef DBIdType FaceId;
typedef DBIdType MasterId;

typedef std::unordered_set<DBIdType> IdSet;
typedef std::set<DBIdType> OrderedIdSet;

typedef std::list<EdgeId> EdgeIdsList;
typedef std::set<EdgeId> EdgeIdsSet;

using EdgeIdsVector = std::vector<EdgeId>;
using FaceIdsVector = std::vector<FaceId>;

typedef uint16_t DispClassId;
typedef uint16_t SearchClassId;

enum class TableName {
    NODE,
    EDGE,
    FACE,
    FACE_EDGE,
    FT_EDGE,
    FT_FACE,
    AD_FACE,
    COUNT
};


class IdGenerator {
public:
    explicit IdGenerator(DBIdType startId) : id_(startId) {}

    DBIdType getId() { return ++id_; }

private:
    std::atomic<DBIdType> id_;
};


class MissingObjectException : public maps::Exception {};

class InvalidFaceException : public maps::Exception {};


const double MERCATOR_EPS = 0.05;


enum class SRID {Mercator, Geodetic};

class UnknownSRIDException : public maps::Exception {};

const std::string STR_MERCATOR = "mercator";
const std::string STR_GEODETIC = "geodetic";

inline SRID sridFromString(const std::string& sridStr)
{
    const std::map<std::string, SRID> strToSRID = {
        {STR_MERCATOR, SRID::Mercator},
        {STR_GEODETIC, SRID::Geodetic}
    };
    auto it = strToSRID.find(sridStr);
    if (it == strToSRID.end()) {
        throw UnknownSRIDException() << "Unknown SRID string: " << sridStr;
    }
    return it->second;
}

inline std::ostream& operator<<(std::ostream& stream, SRID srid)
{
    switch (srid) {
    case SRID::Mercator:
        stream << STR_MERCATOR;
        break;
    case SRID::Geodetic:
        stream << STR_GEODETIC;
        break;
    }
    return stream;
}

#define INSTANTIATE_SET_OPERATION(operationName, stdAlgoName)   \
                                                                \
template <class Container>                                      \
Container                                                       \
operationName(const Container& ids1, const Container& ids2)     \
{                                                               \
    typedef std::set<DBIdType> OrderedIdSet;                    \
    OrderedIdSet orderedIds1(ids1.begin(), ids1.end());         \
    OrderedIdSet orderedIds2(ids2.begin(), ids2.end());         \
    Container result;                                           \
    stdAlgoName(                                                \
        orderedIds1.begin(), orderedIds1.end(),                 \
        orderedIds2.begin(), orderedIds2.end(),                 \
        std::inserter(result, result.end()));                   \
    return result;                                              \
}                                                               \

INSTANTIATE_SET_OPERATION(idsDiff, std::set_difference)
INSTANTIATE_SET_OPERATION(idsSymDiff, std::set_symmetric_difference)
INSTANTIATE_SET_OPERATION(idsIntersection, std::set_intersection)
INSTANTIATE_SET_OPERATION(idsUnion, std::set_union)

#undef INSTANTIATE_SET_OPERATION

template <class IdsContainer>
std::string
toString(const IdsContainer& ids)
{
    std::string result;
    for (auto id : ids) {
        result += (result.empty() ? "" : ", ");
        result += std::to_string(id);
    }
    return result;
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
