#include "faces_merger.h"

#include "../utils/geom_helpers.h"
#include "../utils/searchers.h"

#include <yandex/maps/wiki/threadutils/executor.h>
#include <yandex/maps/wiki/common/batch.h>
#include <maps/libs/common/include/profiletimer.h>

namespace maps {
namespace wiki {
namespace topology_fixer {

namespace {

namespace gl = maps::geolib3;

constexpr size_t BATCH_COUNT = 500;

bool
facesMergingPossible(
    utils::TopologyDataProxy& data, FaceId faceId, FaceId mergedFaceId)
{
    const auto& masterIds = data.face(faceId).masterIds();
    if (masterIds.empty() || masterIds.size() > 1) {
        return false;
    }
    const auto& master = data.master(*masterIds.begin());
    const auto& mergedMasterIds = data.face(mergedFaceId).masterIds();
    if (mergedMasterIds.empty() || mergedMasterIds.size() > 1) {
        return false;
    }
    const auto& mergedMaster = data.master(*mergedMasterIds.begin());
    const auto faceRelType = master.relationType(faceId);
    const auto mergedFaceRelType = mergedMaster.relationType(mergedFaceId);

    if (!((faceRelType == FaceRelationType::Interior &&
          mergedFaceRelType == FaceRelationType::Exterior) ||
         (faceRelType == FaceRelationType::Exterior &&
          mergedFaceRelType == FaceRelationType::Interior)))
    {
        return false;
    }
    return true;
}

} // namespace

FacesMerger::FaceDataMap
FacesMerger::buildFaceData(const TopologyData& data) const
{
    FaceDataMap result;
    for (auto faceId : data.faceIds()) {
        geolib3::PointsVector points;
        try {
            points = data.faceVertices(faceId);
        } catch (const std::exception& e) {
            WARN() << "Face " << faceId << " is invalid";
            continue;
        }
        if (points.empty()) {
            continue;
        }
        result.insert({faceId, {boundingBox(points), points}});
    }
    return result;
}

void
FacesMerger::operator()(ThreadPool& pool)
{
    executeInThreads<FaceIdsVector>(
        pool,
        allFaceIds_,
        BATCH_COUNT,
        [this](const FaceIdsVector& faceIds) {
            ProfileTimer pt;

            INFO() << "FacesMerger start batch with the face count " << faceIds.size();

            for (auto faceId : faceIds) {
                if (processedIds_.has(faceId)) {
                    continue;
                }

                const auto& faceBBox = searcher_.bbox(faceId);
                ASSERT(faceBBox);
                IdSet allIntersectedFaceIds = searcher_.idsIntersectingBBox(*faceBBox);
                processFaces(faceId, allIntersectedFaceIds);
                processedIds_.insert(faceId);
            }

            INFO() << "FacesMerger batched finished in " << pt.getElapsedTime()
                << " face count " << faceIds.size();
        });
}

void
FacesMerger::processFaces(
    FaceId faceId, const IdSet& intersectedFaceIds)
{
    if (!faceData_.count(faceId)) {
        WARN() << "Face " << faceId << " not found";
        return;
    }
    const auto& points = faceData_.at(faceId).points;

    IdSet mergedFaceIds;
    for (auto otherFaceId : intersectedFaceIds) {
        if (faceId == otherFaceId) {
            continue;
        }
        if (processedIds_.has(otherFaceId)) {
            continue;
        }
        if (!faceData_.count(otherFaceId)) {
            WARN() << "Face " << otherFaceId << " not found";
            continue;
        }
        if (!facesMergingPossible(data_, faceId, otherFaceId)) {
            continue;
        }

        const auto& otherPoints = faceData_.at(otherFaceId).points;

        if (areFacesSame(faceId, points, otherFaceId, otherPoints,
            params_.tolerance(),
            params_.maxAngleBetweenSegments(),
            params_.maxTranslationFactor(),
            params_.maxAPRatioFactor()))
        {
            mergedFaceIds.insert(otherFaceId);
        }
    }
    if (mergedFaceIds.empty() || mergedFaceIds.size() > 1) {
        return;
    }
    auto mergedFaceId = *mergedFaceIds.begin();

    auto locks = locker_.lockFaces(std::vector<FaceId>{{faceId, mergedFaceId}});

    if (processedIds_.has(faceId) || processedIds_.has(mergedFaceId)) {
        return;
    }

    // TODO do nothing if edges are already the same
    auto affectedEdgeIds = data_.face(mergedFaceId).edgeIds();
    for (auto edgeId : affectedEdgeIds) {
        data_.removeFaceEdgeRel(mergedFaceId, edgeId);
    }
    for (auto edgeId : data_.face(faceId).edgeIds()) {
        data_.addFaceEdge(mergedFaceId, edgeId);
    }
    for (auto edgeId : affectedEdgeIds) {
        const Edge& e = data_.edge(edgeId);
        if (e.faceIds().empty() && e.masterIds().empty()) {
            data_.removeEdge(edgeId, DeletionMode::Cascade);
        }
    }

    processedIds_.insert(faceId);
    processedIds_.insert(mergedFaceId);
    INFO() << "Merged faces, face id: " << faceId
        << ", removed face ids: " << mergedFaceId;
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
