#include "features_merger.h"

#include "../utils/geom_helpers.h"
#include "../utils/searchers.h"

#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>

#include <initializer_list>

namespace maps {
namespace wiki {
namespace topology_fixer {

namespace {

namespace gl = maps::geolib3;

std::set<FaceId>
masterFaces(utils::TopologyDataProxy& data,
    MasterId masterId, FaceRelationType relationType)
{
    const Master& master = data.master(masterId);
    std::set<FaceId> faceIds;
    for (const auto& faceRel: master.faceRels()) {
        if (faceRel.second == relationType) {
            faceIds.insert(faceRel.first);
        }
    }
    return faceIds;
}

const double MIN_SEARCH_BOX_SIZE = 10.0;

OrderedIdSet
edgeMasters(const TopologyData& data, EdgeId edgeId)
{
    OrderedIdSet masterIds;
    for (auto faceId : data.edge(edgeId).faceIds()) {
        const auto& faceMasters = data.face(faceId).masterIds();
        masterIds.insert(faceMasters.begin(), faceMasters.end());
    }
    return masterIds;
}

bool
diffEmpty(const OrderedIdSet& ids1, const IdSet& ids2)
{
    for (auto id : ids1) {
        if (!ids2.count(id)) {
            return false;
        }
    }
    return true;
}

std::list<OrderedIdSet>
splitEdgeIdsByGrid(const TopologyData& data, size_t gridSize)
{
    std::list<OrderedIdSet> result;
    FaceMasterSearcher searcher(data);
    const auto& bboxOpt = searcher.bbox();
    if (!bboxOpt) {
        return {};
    }
    auto bbox = *bboxOpt;
    const double boxWidth = std::max(bbox.width() / gridSize, MIN_SEARCH_BOX_SIZE);
    const double boxHeight = std::max(bbox.height() / gridSize, MIN_SEARCH_BOX_SIZE);

    for (double minX = bbox.minX(); minX < bbox.maxX(); minX += boxWidth) {
        for (double minY = bbox.minY(); minY < bbox.maxY(); minY += boxHeight) {
            gl::BoundingBox b{{minX, minY}, {minX + boxWidth, minY + boxHeight}};
            b = gl::resizeByValue(b, -MERCATOR_EPS - gl::EPS);
            OrderedIdSet edgeIds;
            const auto& masterIds = searcher.idsByBBox(b);
            for (auto masterId : masterIds) {
                for (const auto& faceRel: data.master(masterId).faceRels()) {
                    for (auto edgeId : data.face(faceRel.first).edgeIds()) {
                        const Edge& edge = data.edge(edgeId);
                        if (edge.faceIds().size() > 1 &&
                            diffEmpty(edgeMasters(data, edgeId), masterIds))
                        {
                            edgeIds.insert(edgeId);
                        }
                    }
                }
            }
            if (!edgeIds.empty()) {
                result.push_back(std::move(edgeIds));
            }
        }
    }
    return result;
}

const size_t GRID_SIZE = 16;


OrderedIdSet
idsDiffOrdered(const OrderedIdSet& ids1, const OrderedIdSet& ids2)
{
    OrderedIdSet result;
    std::set_difference(ids1.begin(), ids1.end(), ids2.begin(), ids2.end(),
        std::inserter(result, result.end()));
    return result;
}

} // namespace


void
FeaturesMerger::operator () (TopologyData& data) const
{
    MasterAreaHolder masterAreas(data);
    OrderedIdSet processedEdgeIds;
    ErrorFlag errorFlag(false);
    size_t gridSize = GRID_SIZE;
    while (gridSize > 1) {
        OrderedIdSet currentProcessedEdgeIds;
        auto edgeBatchesList = splitEdgeIdsByGrid(data, gridSize);
        utils::ThreadSafeTopologyDataProxy dataProxy(data);
        ThreadPool pool(threads_);
        for (const auto& edgeIds : edgeBatchesList) {
            OrderedIdSet unprocessedIds = idsDiffOrdered(edgeIds, processedEdgeIds);
            ASSERT(pool.push(
                EdgesBatch(dataProxy, masterAreas, unprocessedIds,
                    maxMasterFacesCount_, maxFaceEdgesCount_, maxFaceBBoxSize_,
                    errorFlag)));
            REQUIRE(!errorFlag, "Can not continue processing due to error");
            currentProcessedEdgeIds.insert(edgeIds.begin(), edgeIds.end());
        }
        pool.shutdown();
        processedEdgeIds.insert(
            currentProcessedEdgeIds.begin(), currentProcessedEdgeIds.end());
        gridSize /= 2;
    }

    OrderedIdSet remainedEdgeIds;
    auto edgeIdsList = data.edgeIds();
    std::remove_copy_if(
        edgeIdsList.begin(), edgeIdsList.end(),
        std::inserter(remainedEdgeIds, remainedEdgeIds.end()),
        [&] (EdgeId edgeId) -> bool
        {
            return processedEdgeIds.count(edgeId) ||
                data.edge(edgeId).faceIds().size() == 1;
        });

    REQUIRE(!errorFlag, "Can not continue processing due to error");
    utils::TrivialTopologyDataProxy dataProxy(data);
    EdgesBatch(
        dataProxy, masterAreas, remainedEdgeIds,
        maxMasterFacesCount_, maxFaceEdgesCount_, maxFaceBBoxSize_,
        errorFlag)
    ();
    REQUIRE(!errorFlag, "Can not continue processing due to error");
}

FeaturesMerger::EdgesBatch::EdgesBatch(
        utils::TopologyDataProxy& data,
        MasterAreaHolder& masterAreas,
        const OrderedIdSet& edgeIds,
        size_t maxMasterFacesCount,
        size_t maxFaceEdgesCount,
        double maxFaceBBoxSize,
        ErrorFlag& errorFlag)
    : data_(data)
    , masterAreas_(masterAreas)
    , edgeIds_(edgeIds)
    , maxMasterFacesCount_(maxMasterFacesCount)
    , maxFaceEdgesCount_(maxFaceEdgesCount)
    , maxFaceBBoxSize_(maxFaceBBoxSize)
    , errorFlag_(errorFlag)
{}

namespace {

struct SharedEdgeData {
    FaceId leftMasterFaceId;
    MasterId leftMasterId;
    FaceId removedMasterFaceId;
    MasterId removedMasterId;
};

boost::optional<SharedEdgeData>
parseEdgeData(utils::TopologyDataProxy& data, EdgeId edgeId)
{
    const Edge& edge = data.edge(edgeId);
    if (edge.faceIds().size() != 2) {
        return boost::none;
    }
    const FaceId faceId1 = *edge.faceIds().begin();
    const FaceId faceId2 = *std::next(edge.faceIds().begin());
    const IdSet& masterIds1 = data.face(faceId1).masterIds();
    const IdSet& masterIds2 = data.face(faceId2).masterIds();
    if (masterIds1.size() != 1 || masterIds2.size() != 1) {
        return boost::none;
    }
    const MasterId masterId1 = *masterIds1.begin();
    const MasterId masterId2 = *masterIds2.begin();

    if (masterId1 == masterId2) {
        return boost::none;
    }

    const Master master1 = data.master(masterId1);
    const Master master2 = data.master(masterId2);
    if (!master1.ftAttributes() || !master2.ftAttributes()) {
        return boost::none;
    }
    const FtAttributes& ftAttrs1 = *master1.ftAttributes();
    const FtAttributes& ftAttrs2 = *master2.ftAttributes();
    if (master1.relationType(faceId1) == FaceRelationType::Exterior &&
        masterFaces(data, masterId1, FaceRelationType::Exterior).size() == 1 &&
        ftAttrs1.canBeMergedWith(ftAttrs2))
    {
        return SharedEdgeData {faceId2, masterId2, faceId1, masterId1};
    }
    if (master2.relationType(faceId2) == FaceRelationType::Exterior &&
        masterFaces(data, masterId2, FaceRelationType::Exterior).size() == 1 &&
        ftAttrs2.canBeMergedWith(ftAttrs1))
    {
        return SharedEdgeData {faceId1, masterId1, faceId2, masterId2};
    }

    return boost::none;
}

bool
areSharedEdgeAndNodesValid(
    utils::TopologyDataProxy& data, const SharedEdgeData& sharedEdgeData)
{
    IdSet faceIds =
        {sharedEdgeData.leftMasterFaceId, sharedEdgeData.removedMasterFaceId};

    const auto& edgeIds1 = data.face(sharedEdgeData.leftMasterFaceId).edgeIds();
    const auto& edgeIds2 = data.face(sharedEdgeData.removedMasterFaceId).edgeIds();
    const auto& sharedEdgeIds = idsIntersection(edgeIds1, edgeIds2);
    IdSet sharedNodeIds;
    for (auto sEdgeId : sharedEdgeIds) {
        const Edge& edge = data.edge(sEdgeId);
        const auto& edgeFaces = edge.faceIds();
        if (!idsDiff(edgeFaces, faceIds).empty()) {
            return false;
        }
        sharedNodeIds.insert(edge.fnodeId());
        sharedNodeIds.insert(edge.tnodeId());
    }
    for (auto sNodeId : sharedNodeIds) {
        const Node& node = data.node(sNodeId);
        const auto& nodeEdges = node.edgeIds();
        if (idsIntersection(nodeEdges, sharedEdgeIds).size() > 1 &&
            nodeEdges.size() != 2)
        {
            return false;
        }
    }
    return true;
}

/**
 * Copy
 *   1) both masters
 *   2) all faces of deleted master, affected face of united master
 *   3) all face edges and their nodes
 *   4) all edges incident with nodes that are incident with all shared edges of
 *     both faces
 * Result can be null due to following reasons:
 *  1) any of shared edges in connected to some other face
 *  2) any of internal nodes from chain of shared edges has any other incident edge
 */
std::unique_ptr<TopologyData>
copyData(utils::TopologyDataProxy& data, const SharedEdgeData& sharedEdgeData)
{
    std::list<NodeData> nodes;
    std::list<EdgeData> edges;
    std::list<FaceEdgeData> faceEdgeRels;
    std::list<FaceMasterData> masterFaceRels;

    const Master& leftMaster = data.master(sharedEdgeData.leftMasterId);
    const Master& removedMaster = data.master(sharedEdgeData.removedMasterId);
    REQUIRE(removedMaster.edgeIds().empty(),
        "Master " << removedMaster.id() << " is not contour only object");
    for (const auto& faceRel: leftMaster.faceRels()) {
        masterFaceRels.push_back(
            {leftMaster.id(), faceRel.first, faceRel.second});
    }
    for (const auto& faceRel: removedMaster.faceRels()) {
        masterFaceRels.push_back(
            {removedMaster.id(), faceRel.first, faceRel.second});
    }

    if (!areSharedEdgeAndNodesValid(data, sharedEdgeData)) {
        return nullptr;
    }

    IdSet faceIds;
    for (const auto& faceRel: removedMaster.faceRels()) {
        faceIds.insert(faceRel.first);
    }
    for (const auto& faceRel: leftMaster.faceRels()) {
        faceIds.insert(faceRel.first);
    }

    IdSet allEdgeIds;
    for (auto faceId : faceIds) {
        const auto& faceEdges = data.face(faceId).edgeIds();
        for (auto edgeId : faceEdges) {
            faceEdgeRels.push_back({faceId, edgeId});
        }
        allEdgeIds.insert(faceEdges.begin(), faceEdges.end());
    }
    IdSet allNodeIds;
    for (auto edgeId : allEdgeIds) {
        const Edge& edge = data.edge(edgeId);
        edges.push_back({edgeId, edge.linestring(),
            edge.fnodeId(), edge.tnodeId(),
            edge.fZlev(), edge.tZlev()});
        allNodeIds.insert(edge.fnodeId());
        allNodeIds.insert(edge.tnodeId());
    }
    for (auto nodeId : allNodeIds) {
        const Node& node = data.node(nodeId);
        nodes.push_back({nodeId, node.point()});
    }

    std::unique_ptr<TopologyData> dataCopy(new TopologyData(
        data.ftGroup(),
        nodes, edges, faceEdgeRels, std::list<EdgeMasterData>(), masterFaceRels,
        data.idGenerator(), data.srid()));
    dataCopy->setFtAttributes(removedMaster.id(), *removedMaster.ftAttributes());
    dataCopy->setFtAttributes(leftMaster.id(), *leftMaster.ftAttributes());

    return dataCopy;
}

typedef std::list<IdSet> SetList;

SetList
splitNewFaceEdges(const TopologyData& dataCopy, const IdSet& unsharedEdgeIds)
{
    SetList newFaceEdges;
    SetList newFaceNodes;
    for (auto edgeId : unsharedEdgeIds) {
        const Edge& edge = dataCopy.edge(edgeId);
        std::list<SetList::iterator> mergedNodeSets;
        std::list<SetList::iterator> mergedEdgeSets;
        auto nit = newFaceNodes.begin();
        auto eit = newFaceEdges.begin();
        for ( ; nit != newFaceNodes.end(); ++nit, ++eit) {
            if (nit->count(edge.fnodeId()) || nit->count(edge.tnodeId())) {
                mergedNodeSets.push_back(nit);
                mergedEdgeSets.push_back(eit);
            }
        }
        IdSet nodeIds;
        IdSet edgeIds;
        for (auto it : mergedNodeSets) {
            nodeIds.insert(it->begin(), it->end());
            newFaceNodes.erase(it);
        }
        nodeIds.insert(edge.fnodeId());
        nodeIds.insert(edge.tnodeId());
        for (auto it : mergedEdgeSets) {
            edgeIds.insert(it->begin(), it->end());
            newFaceEdges.erase(it);
        }
        edgeIds.insert(edgeId);
        newFaceNodes.push_back(std::move(nodeIds));
        newFaceEdges.push_back(std::move(edgeIds));
    }
    return newFaceEdges;
}

void
rebuildRelationsExceptMergedFaces(
    TopologyData& dataCopy, const SharedEdgeData& sharedEdgeData)
{
    const MasterId mergedMasterId = sharedEdgeData.leftMasterId;
    const FaceId mergedMasterFaceId = sharedEdgeData.leftMasterFaceId;
    const MasterId removedMasterId = sharedEdgeData.removedMasterId;
    const FaceId removedMasterFaceId = sharedEdgeData.removedMasterFaceId;

    IdSet mergedEdgeIds = dataCopy.face(mergedMasterFaceId).edgeIds();
    IdSet removedEdgeIds = dataCopy.face(removedMasterFaceId).edgeIds();
    IdSet sharedEdgeIds = idsIntersection(mergedEdgeIds, removedEdgeIds);

    for (auto edgeId : mergedEdgeIds) {
        dataCopy.removeFaceEdgeRel(mergedMasterFaceId, edgeId);
    }
    for (auto edgeId : removedEdgeIds) {
        dataCopy.removeFaceEdgeRel(removedMasterFaceId, edgeId);
    }

    for (auto edgeId : sharedEdgeIds) {
        dataCopy.removeEdge(edgeId, DeletionMode::Cascade);
    }

    dataCopy.removeMasterFaceRel(mergedMasterId, mergedMasterFaceId);
    dataCopy.removeMasterFaceRel(removedMasterId, removedMasterFaceId);
    dataCopy.removeFace(mergedMasterFaceId, DeletionMode::Restrict);
    dataCopy.removeFace(removedMasterFaceId, DeletionMode::Restrict);

    for (const auto& faceRel: dataCopy.master(removedMasterId).faceRels())
    {
        dataCopy.addMasterFaceRel(mergedMasterId, faceRel.first, faceRel.second);
    }
    for (auto faceId : dataCopy.master(removedMasterId).faceIds()) {
        dataCopy.removeMasterFaceRel(removedMasterId, faceId);
    }

    dataCopy.removeMaster(removedMasterId, DeletionMode::Restrict);
}

typedef std::pair<FaceRelations, bool> FaceBuildResult;

FaceBuildResult
buildAndValidateNewFaces(
    TopologyData& dataCopy,
    const SetList& newFaceEdges,
    FaceRelationType mergedMasterFaceRelType)
{
    if (newFaceEdges.empty()) {
        return {{}, mergedMasterFaceRelType == FaceRelationType::Interior};
    }

    IdSet newFaceIds;
    for (const auto& edgeIds : newFaceEdges) {
        auto newFaceId = dataCopy.addFace();
        for (auto edgeId : edgeIds) {
            dataCopy.addFaceEdge(newFaceId, edgeId);
        }
        newFaceIds.insert(newFaceId);
    }
    std::map<FaceId, geolib3::BoundingBox> newFaceBBoxes;
    for (auto faceId : newFaceIds) {
        try {
            const auto& points = dataCopy.faceVertices(faceId);
            geolib3::Polygon2 poly(points);
            newFaceBBoxes.insert({faceId, poly.boundingBox()});
        } catch (const std::exception& e) {
            return {{}, false};
        }
    }

    FaceRelations faceRelationTypes;

    if (mergedMasterFaceRelType == FaceRelationType::Interior) {
        for (auto faceId : newFaceIds) {
            faceRelationTypes.insert({faceId, FaceRelationType::Interior});
        }
        return {faceRelationTypes, true};
    }

    auto it = std::max_element(
        newFaceBBoxes.begin(), newFaceBBoxes.end(),
        [&] (const std::pair<FaceId, geolib3::BoundingBox>& lhs,
             const std::pair<FaceId, geolib3::BoundingBox>& rhs) -> bool
        {
            return lhs.second.width() * lhs.second.height() <
                rhs.second.width() * rhs.second.height();
        });
    ASSERT(it != newFaceBBoxes.end());
    const FaceId maxFaceId = it->first;
    for (auto faceId : newFaceIds) {
        faceRelationTypes.insert({
            faceId,
            (faceId == maxFaceId
                ? FaceRelationType::Exterior
                : FaceRelationType::Interior)});
    }
    return {faceRelationTypes, true};
}

bool
areRestrictionsSatisfied(
    const TopologyData& dataCopy,
    MasterId masterId,
    const SetList& newFaceEdges,
    size_t maxMasterFaces, size_t maxFaceEdges, double maxFaceBBoxSize)
{
    const Master& master = dataCopy.master(masterId);
    const size_t origFacesCount = master.faceIds().size();
    REQUIRE(origFacesCount > 0, "Master " << masterId << " has no geom");
    if (newFaceEdges.size() + origFacesCount - 1 > maxMasterFaces) {
        return false;
    }
    for (const auto& edgeIds : newFaceEdges) {
        if (edgeIds.size() > maxFaceEdges) {
            return false;
        }
        boost::optional<geolib3::BoundingBox> faceBBox;
        for (auto edgeId : edgeIds) {
            const auto& edgeBBox = dataCopy.edge(edgeId).boundingBox();
            faceBBox = faceBBox ? geolib3::expand(*faceBBox, edgeBBox) : edgeBBox;
        }
        if (faceBBox && std::max(faceBBox->width(), faceBBox->height()) > maxFaceBBoxSize) {
            return false;
        }
    }

    return true;
}

const double AREA_DIFF_FACTOR = 1e-4;
const double MIN_AREA = 1.0;

bool
areasEquivalent(const Area& area1, const Area& area2, const Area& resultArea)
{
    double a1 = area1.exterior - area1.interior;
    double a2 = area2.exterior - area2.interior;
    double a = resultArea.exterior - resultArea.interior;
    if (a < MIN_AREA) {
        return false;
    }

    return std::fabs(a1 + a2 - a) / a < AREA_DIFF_FACTOR;
}

bool
facesTouch(const TopologyData& dataCopy, const SharedEdgeData& sharedEdgeData)
{
    const MasterId mergedMasterId = sharedEdgeData.leftMasterId;
    const FaceId mergedMasterFaceId = sharedEdgeData.leftMasterFaceId;
    const FaceId removedMasterFaceId = sharedEdgeData.removedMasterFaceId;
    const FaceRelationType mergedMasterFaceRelType =
        dataCopy.master(mergedMasterId).relationType(mergedMasterFaceId);
    try {
        const auto removedFacePoints = dataCopy.faceVertices(removedMasterFaceId);
        const auto mergedFacePoints = dataCopy.faceVertices(mergedMasterFaceId);
        geolib3::Polygon2 removedPoly(removedFacePoints);
        geolib3::Polygon2 mergedPoly;
        if (mergedMasterFaceRelType == FaceRelationType::Exterior) {
            mergedPoly = geolib3::Polygon2(mergedFacePoints);
        } else {
            REQUIRE(!mergedFacePoints.empty(), "Empty mergedFacePoints");
            auto bbox = boundingBox(mergedFacePoints);
            bbox = geolib3::resizeByRatio(bbox, 2.0);
            geolib3::LinearRing2 ring(geolib3::PointsVector{
                bbox.lowerCorner(), geolib3::Point2{bbox.maxX(), bbox.minY()},
                bbox.upperCorner(), geolib3::Point2{bbox.minX(), bbox.maxY()},
                bbox.lowerCorner()});
            mergedPoly = geolib3::Polygon2(ring, {geolib3::LinearRing2(mergedFacePoints)});
        }
        return geolib3::spatialRelation(
            mergedPoly, removedPoly, geolib3::SpatialRelation::Touches);
    } catch (const std::exception&)
    {}
    return false;
}

struct FaceReconstructionResult {
    bool valid;
    bool needsAdditionalCheck;
};

FaceReconstructionResult
rebuildFaces(TopologyData& dataCopy, const SharedEdgeData& sharedEdgeData,
    size_t maxMasterFaces, size_t maxFaceEdges, double maxFaceBBoxSize)
{
    const MasterId mergedMasterId = sharedEdgeData.leftMasterId;
    const FaceId mergedMasterFaceId = sharedEdgeData.leftMasterFaceId;
    const FaceId removedMasterFaceId = sharedEdgeData.removedMasterFaceId;
    const FaceRelationType mergedMasterFaceRelType =
        dataCopy.master(mergedMasterId).relationType(mergedMasterFaceId);

    const auto& mergedEdgeIds = dataCopy.face(mergedMasterFaceId).edgeIds();
    const auto& removedEdgeIds = dataCopy.face(removedMasterFaceId).edgeIds();
    const auto& unsharedEdgeIds = idsSymDiff(mergedEdgeIds, removedEdgeIds);

    if (mergedMasterFaceRelType == FaceRelationType::Interior &&
        unsharedEdgeIds.empty())
    {
        rebuildRelationsExceptMergedFaces(dataCopy, sharedEdgeData);
        return {true, false};
    }

    SetList newFaceEdges = splitNewFaceEdges(dataCopy, unsharedEdgeIds);

    if (mergedMasterFaceRelType == FaceRelationType::Exterior &&
        !areRestrictionsSatisfied(
            dataCopy, sharedEdgeData.leftMasterId, newFaceEdges,
            maxMasterFaces, maxFaceEdges, maxFaceBBoxSize))
    {
        return {false, false};
    }

    rebuildRelationsExceptMergedFaces(dataCopy, sharedEdgeData);

    FaceBuildResult result = buildAndValidateNewFaces(
        dataCopy, newFaceEdges, mergedMasterFaceRelType);

    if (result.second) {
        for (const auto& faceRel : result.first) {
            dataCopy.addMasterFaceRel(mergedMasterId, faceRel.first, faceRel.second);
        }
    }

    return {result.second, true};
}

} // namespace

void
FeaturesMerger::EdgesBatch::doWork()
{
    IdSet processedEdgeIds;
    for (auto edgeId : edgeIds_) {
        if (processedEdgeIds.count(edgeId)) {
            continue;
        }
        boost::optional<SharedEdgeData> sharedEdgeData = parseEdgeData(data_, edgeId);
        if (!sharedEdgeData) {
            continue;
        }

        auto sharedEdgeIds = idsIntersection(
            data_.face(sharedEdgeData->leftMasterFaceId).edgeIds(),
            data_.face(sharedEdgeData->removedMasterFaceId).edgeIds());

        std::unique_ptr<TopologyData> dataCopy = copyData(data_, *sharedEdgeData);
        if (!dataCopy) {
            processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
            continue;
        }
        std::unique_ptr<TopologyData> originalDataCopy = dataCopy->clone();
        FaceReconstructionResult rebuildResult = rebuildFaces(
            *dataCopy, *sharedEdgeData,
            maxMasterFacesCount_, maxFaceEdgesCount_, maxFaceBBoxSize_);
        if (!rebuildResult.valid)
        {
            processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
            continue;
        }

        boost::optional<Area> resultArea = masterArea(
            *dataCopy, sharedEdgeData->leftMasterId);
        if (!resultArea) {
            processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
            continue;
        }
        if (rebuildResult.needsAdditionalCheck) {
            boost::optional<Area> removedArea =
                masterAreas_.area(sharedEdgeData->removedMasterId);
            boost::optional<Area> mergedArea =
                masterAreas_.area(sharedEdgeData->leftMasterId);
            if (!removedArea || !mergedArea) {
                processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
                continue;
            }
            if (!resultArea ||
                !areasEquivalent(*removedArea, *mergedArea, *resultArea) ||
                !facesTouch(*originalDataCopy, *sharedEdgeData))
            {
                processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
                continue;
            }
        }

        masterAreas_.setArea(sharedEdgeData->leftMasterId, *resultArea);
        TopologyDataDiff diff = computeDiff(*dataCopy, *originalDataCopy);
        applyDiff(data_, *dataCopy, diff);
        INFO() << "Master " << sharedEdgeData->leftMasterId
            << ", face " << sharedEdgeData->leftMasterFaceId << " merged with "
            << " master " << sharedEdgeData->removedMasterId
            << ", face " << sharedEdgeData->removedMasterFaceId << " (removed)";
        processedEdgeIds.insert(sharedEdgeIds.begin(), sharedEdgeIds.end());
    }
}

} // namespace topology_fixer
} // namespace wiki
} // namespace maps
